/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*                  This file is part of the program and library             */
/*          GCG --- Generic Column Generation                                */
/*                  a Dantzig-Wolfe decomposition based extension            */
/*                  of the branch-cut-and-price framework                    */
/*         SCIP --- Solving Constraint Integer Programs                      */
/*                                                                           */
/* Copyright (C) 2010-2026 Operations Research, RWTH Aachen University       */
/*                         Zuse Institute Berlin (ZIB)                       */
/*                                                                           */
/*  Licensed under the Apache License, Version 2.0 (the "License");          */
/*  you may not use this file except in compliance with the License.         */
/*  You may obtain a copy of the License at                                  */
/*                                                                           */
/*      http://www.apache.org/licenses/LICENSE-2.0                           */
/*                                                                           */
/*  Unless required by applicable law or agreed to in writing, software      */
/*  distributed under the License is distributed on an "AS IS" BASIS,        */
/*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. */
/*  See the License for the specific language governing permissions and      */
/*  limitations under the License.                                           */
/*                                                                           */
/*  You should have received a copy of the Apache-2.0 license                */
/*  along with GCG; see the file LICENSE. If not visit gcg.or.rwth-aachen.de.*/
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file        event_mastersepacut.c
 * @ingroup     DEFPLUGINS_EVENT
 * @brief       event handler which manages the active master separator cuts
 * @author      Chantal Reinartz Groba
 */

/*--+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/
//#define SCIP_DEBUG

#include "scip/struct_scip.h"
#include "scip/type_rational.h"
#include "scip/struct_tree.h"
#include "gcg/event_mastersepacut.h"
#include "gcg/gcg.h"
#include "gcg/mastersepacut.h"
#include "gcg/pricer_gcg.h"
#include "gcg/struct_gcg.h"


#define EVENTHDLR_NAME         "mastersepacut"
#define EVENTHDLR_DESC         "event handler for keeping master cuts generated by a separators up-to-date"
#define STARTMAXCUTS            50

/*
 * Data structures
 */


/** event handler data */
struct SCIP_EventhdlrData
{
   GCG*                          gcg;                    /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA**  generatedcuts;          /**< arrays of newly generated mastersepacuts: corresponding rows were added to sepa store */
   GCG_EXTENDEDMASTERCONSDATA**  activecuts;             /**< arrays of active mastersepacuts: corresponding rows are part of the current LP */
   SCIP_HASHMAP*                 rowxgeneratedmap;       /**< maps a row to the index of its mastersepacut in generated cuts */
   int                           nactivecuts;            /**< number of active mastersepacuts */
   int                           ngeneratedcuts;         /**< number of newly generated mastersepacuts */
   int                           activecutssize;         /**< size of memory allocated in active cuts */
   int                           generatedcutssize;      /**< size of memory allocated in generated cuts */
   SCIP_Bool                     globalcutsreinserted;   /**< have global cuts been re-inserted */
};

/*
 * Local methods
 */

/** ensures that generated cuts has enough allocated memory to store specified number of elements */
static
SCIP_RETCODE ensureGeneratedSize(
   SCIP*                masterscip,             /**< SCIP data structure (master problem) */
   SCIP_EVENTHDLRDATA*  eventhdlrdata,          /**< event handler data */
   int                  size                    /**< number of elements generated cuts should be able to hold */
   )
{
   assert(masterscip != NULL);
   assert(eventhdlrdata != NULL);
   assert(eventhdlrdata->generatedcuts != NULL);
   assert(eventhdlrdata->generatedcuts != NULL);
   assert(eventhdlrdata->ngeneratedcuts <= eventhdlrdata->generatedcutssize);
   assert(eventhdlrdata->ngeneratedcuts >= 0);

   if( eventhdlrdata->generatedcutssize < size )
   {
      int newmaxcuts = SCIPcalcMemGrowSize(masterscip, size);
      SCIP_CALL( SCIPreallocBlockMemoryArray(masterscip, &(eventhdlrdata->generatedcuts), eventhdlrdata->generatedcutssize,
                                             newmaxcuts) );
      eventhdlrdata->generatedcutssize = newmaxcuts;
   }

   assert(eventhdlrdata->generatedcutssize >= size);

   return SCIP_OKAY;
}

/** ensures that active cuts has enough allocated memory to store specified number of elements */
static
SCIP_RETCODE ensureActiveSize(
   SCIP*                   masterscip,         /**< SCIP data structure */
   SCIP_EVENTHDLRDATA*     eventhdlrdata,      /**< event handler data data structure */
   int                     size                /**< number of elements active cuts should be able to hold */
   )
{
   assert(masterscip != NULL);
   assert(eventhdlrdata != NULL);
   assert(eventhdlrdata->activecuts != NULL);
   assert(eventhdlrdata->nactivecuts <= eventhdlrdata->activecutssize);
   assert(eventhdlrdata->nactivecuts >= 0);

   if( eventhdlrdata->activecutssize < size )
   {
      int newmaxcuts = SCIPcalcMemGrowSize(masterscip, size);
      SCIP_CALL( SCIPreallocBlockMemoryArray(masterscip, &(eventhdlrdata->activecuts), eventhdlrdata->activecutssize,
                                             newmaxcuts) );
      eventhdlrdata->activecutssize = newmaxcuts;
   }

   assert(eventhdlrdata->activecutssize >= size);

   return SCIP_OKAY;
}

static
SCIP_RETCODE addToActiveMastersepacuts(
   GCG*                          gcg,                  /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut,        /**< global master separator cut */
   SCIP_EVENTHDLRDATA*           eventhdlrdata         /**< event handler data */
   )
{
   assert(GCGextendedmasterconsGetType(mastersepacut) == GCG_EXTENDEDMASTERCONSTYPE_SEPA_ROW);
   assert(GCGextendedmasterconsGetRow(mastersepacut) != NULL);
   SCIP_CALL( ensureActiveSize(GCGgetMasterprob(gcg), eventhdlrdata, eventhdlrdata->nactivecuts + 1) );
   eventhdlrdata->activecuts[eventhdlrdata->nactivecuts] = mastersepacut;
   SCIP_CALL( GCGcaptureMastersepacut(mastersepacut) );
   (eventhdlrdata->nactivecuts)++;
   return SCIP_OKAY;
}

/** re-inserts the mastersepacut of a global row (in case it was re-added to the separation storage) to generated cuts */
static
SCIP_RETCODE reinsertGlobalMastersepacut(
   GCG*                          gcg,                  /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut,        /**< global master separator cut */
   SCIP_EVENTHDLRDATA*           eventhdlrdata         /**< event handler data */
   )
{
   SCIP_ROW*          row;

   assert(gcg != NULL);
   assert(mastersepacut != NULL);
   assert(eventhdlrdata != NULL);

   row = GCGextendedmasterconsGetRow(mastersepacut);
   assert(!SCIProwIsLocal(row));

   SCIPdebugMessage("re-add global cuts: row %s is added to generated guts\n", SCIProwGetName(row));
   SCIP_CALL( ensureGeneratedSize(GCGgetMasterprob(gcg), eventhdlrdata, eventhdlrdata->ngeneratedcuts + 1) );
   eventhdlrdata->generatedcuts[eventhdlrdata->ngeneratedcuts] = mastersepacut;
   SCIP_CALL( GCGcaptureMastersepacut(eventhdlrdata->generatedcuts[eventhdlrdata->ngeneratedcuts]) );
   SCIP_CALL( SCIPhashmapSetImageInt(eventhdlrdata->rowxgeneratedmap, row, eventhdlrdata->ngeneratedcuts) );
   (eventhdlrdata->ngeneratedcuts)++;

   return SCIP_OKAY;
}

/**< transfers mastersepacut from generatedcuts to activecuts when the corresponding row is added to the LP */
static
SCIP_RETCODE eventRowAddedToLP(
   GCG*                    gcg,              /**< GCG data structure */
   SCIP_EVENTHDLRDATA*     eventhdlrdata,    /**< mastersepacut eventhandler */
   SCIP_EVENT*             event             /**< row added to LP event */
   )
{
   SCIP* scip;
   GCG_EXTENDEDMASTERCONSDATA* generatedcut;
   SCIP_SEPA* sepa;
   SCIP_ROW* row;
   int sepaidx;
   int generatedidx;

   assert(gcg != NULL);
   assert(eventhdlrdata != NULL);
   assert(event != NULL);
   assert(SCIPeventGetType(event) == SCIP_EVENTTYPE_ROWADDEDLP);

   scip = GCGgetMasterprob(gcg);

   /* row is added when LP is already constructed --> row is newly generated cut (or re-added global cut)*/
   if( SCIPisLPConstructed(scip) )
   {
      row = SCIPeventGetRow(event);
      sepa = SCIProwGetOriginSepa(row);

      /* if row was not generated by separator: nothing to do */
      if( sepa == NULL )
      {
         SCIPdebugMessage("row added event: cut was not generated by separator\n");
         return SCIP_OKAY;
      }

      generatedidx = SCIPhashmapGetImageInt(eventhdlrdata->rowxgeneratedmap, row);
      if( generatedidx != INT_MAX )
      {
         SCIPdebugMessage("row added event: row %s is added to active cuts of separator %s (%i) at index %i\n",
                          SCIProwGetName(row), SCIPsepaGetName(sepa), sepaidx, eventhdlrdata->nactivecuts);

         /* transfer cut from generated to active */
         generatedcut = eventhdlrdata->generatedcuts[generatedidx];
         assert(GCGextendedmasterconsGetRow(generatedcut) == row);
         SCIP_CALL( ensureActiveSize(scip, eventhdlrdata, eventhdlrdata->nactivecuts + 1) );

         /* cut is newly generated (not re-added global cut): set its initial variable history */
         if( GCGmastersepacutGetVarHistory(GCGextendedmasterconsGetMastersepacut(generatedcut)) == NULL )
            SCIP_CALL( GCGmastersepacutSetVarHistory(gcg, GCGextendedmasterconsGetMastersepacut(generatedcut)) );

         addToActiveMastersepacuts(gcg, generatedcut, eventhdlrdata);
      }
   }

   return SCIP_OKAY;
}



/*
 * Callback methods of event handler
 */


/** destructor of event handler to free user data (called when SCIP is exiting) */
static
SCIP_DECL_EVENTFREE(eventFreeMastersepacutUpdate)
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(scip != NULL);
   assert(eventhdlr != NULL);
   assert(strcmp(SCIPeventhdlrGetName(eventhdlr), EVENTHDLR_NAME) == 0);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   SCIPdebugMessage("event free: free hashmaps\n");
   SCIPhashmapFree(&(eventhdlrdata->rowxgeneratedmap));

   SCIPdebugMessage("event free: free event handler data\n");
   SCIPfreeBlockMemory(scip, &eventhdlrdata);

   return SCIP_OKAY;
}

static
SCIP_DECL_EVENTEXIT(eventExitMastersepacutUpdate)
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(scip != NULL);
   assert(eventhdlr != NULL);
   assert(strcmp(SCIPeventhdlrGetName(eventhdlr), EVENTHDLR_NAME) == 0);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   /* notify SCIP that your event handler wants to drop the event type 'row added to lp' */
   SCIP_CALL( SCIPdropEvent(scip, SCIP_EVENTTYPE_ROWADDEDLP, eventhdlr, NULL, -1) );

   /* free all the arrays */
   SCIPdebugMessage("event free: free mem (%i) for activecuts\n", eventhdlrdata->activecutssize);
   SCIPfreeBlockMemoryArrayNull(scip, &(eventhdlrdata->activecuts), eventhdlrdata->activecutssize);
   SCIPdebugMessage("event free: free mem (%i) for generatedcuts\n", eventhdlrdata->generatedcutssize);
   SCIPfreeBlockMemoryArrayNull(scip, &(eventhdlrdata->generatedcuts), eventhdlrdata->generatedcutssize);

   eventhdlrdata->nactivecuts = 0;
   eventhdlrdata->ngeneratedcuts = 0;
   eventhdlrdata->generatedcutssize = 0;
   eventhdlrdata->activecutssize = 0;

   return SCIP_OKAY;
}

/** initialization method of event handler (called after problem was transformed) */
static
SCIP_DECL_EVENTINIT(eventInitMastersepacutUpdate)
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   int initialsize;

   assert(scip != NULL);
   assert(GCGisMaster(scip));
   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   /* initialize event handler data */
   SCIPdebugMessage("event init: alloc mem for active cuts and generated cuts\n");
   initialsize = SCIPcalcMemGrowSize(scip, STARTMAXCUTS);
   SCIP_CALL( SCIPallocBlockMemoryArray(scip, &(eventhdlrdata->activecuts), initialsize) );
   SCIP_CALL( SCIPallocBlockMemoryArray(scip, &(eventhdlrdata->generatedcuts), initialsize) );
   eventhdlrdata->nactivecuts = 0;
   eventhdlrdata->ngeneratedcuts = 0;
   eventhdlrdata->activecutssize = initialsize;
   eventhdlrdata->generatedcutssize = initialsize;
   eventhdlrdata->globalcutsreinserted = FALSE;

   /* notify SCIP that event handler wants to react on the event typ 'row added to LP' */
   SCIP_CALL( SCIPcatchEvent(scip, SCIP_EVENTTYPE_ROWADDEDLP, eventhdlr, NULL, NULL) );

   return SCIP_OKAY;
}

/** deinitialization method of event handler (called before transformed problem is freed)
 * - clean up branch-and-bound specific data */
static
SCIP_DECL_EVENTEXITSOL(eventExitSolMastersepacutUpdate)
{
   GCG* gcg;
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   int j;

   assert(scip != NULL);
   assert(GCGisMaster(scip));
   assert(eventhdlr != NULL);
   assert(strcmp(SCIPeventhdlrGetName(eventhdlr), EVENTHDLR_NAME) == 0);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   gcg = eventhdlrdata->gcg;

   /* release all the remaining cuts from generated cuts and active cuts */
   SCIPdebugMessage("event exit sol: release remaining %i cuts in generatedcuts\n", eventhdlrdata->ngeneratedcuts);
   for( j = 0; j < eventhdlrdata->ngeneratedcuts; j++ )
   {
      assert(eventhdlrdata->generatedcuts[j] != NULL);
      SCIP_CALL( GCGreleaseMastersepacut(gcg, &(eventhdlrdata->generatedcuts[j])) );
      assert(eventhdlrdata->generatedcuts[j] == NULL);
   }
   eventhdlrdata->ngeneratedcuts = 0;

   SCIPdebugMessage("event exit sol: release remaining %i cuts in activecuts\n", eventhdlrdata->nactivecuts);
   for( j = 0; j < eventhdlrdata->nactivecuts; j++ )
   {
      assert(eventhdlrdata->activecuts[j] != NULL);
      SCIP_CALL( GCGreleaseMastersepacut(gcg, &(eventhdlrdata->activecuts[j])) );
      assert(eventhdlrdata->activecuts[j] == NULL);
   }
   eventhdlrdata->nactivecuts = 0;

   SCIPdebugMessage("event exit sol: clear hashmap\n");
   SCIPhashmapRemoveAll(eventhdlrdata->rowxgeneratedmap);

   return SCIP_OKAY;
}

/** main execution method of event handler */
static
SCIP_DECL_EVENTEXEC(eventExecEvent)
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(scip != NULL);
   assert(GCGisMaster(scip));
   assert(eventhdlr != NULL);
   assert(strcmp(SCIPeventhdlrGetName(eventhdlr), EVENTHDLR_NAME) == 0);
   assert(event != NULL);
   assert(SCIPeventGetType(event) == SCIP_EVENTTYPE_ROWADDEDLP);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert( eventhdlrdata != NULL );

   switch( SCIPeventGetType(event) )
   {
      case SCIP_EVENTTYPE_ROWADDEDLP:
         SCIP_CALL( eventRowAddedToLP(eventhdlrdata->gcg, eventhdlrdata, event) );
         break;
      default:
         SCIPerrorMessage("Encountered Event not listened to.\n");
         return SCIP_ERROR;
   }

   return SCIP_OKAY;
}

/** creates event handler for managing the additional data of the mastercuts generated by master separators */
SCIP_RETCODE GCGincludeEventHdlrSepaCuts(
   GCG*                  gcg                 /**< GCG data structure */
   )
{
   SCIP* scip;
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   SCIP_EVENTHDLR* eventhdlr;

   scip = GCGgetMasterprob(gcg);

   /* create event handler data */
   SCIPdebugMessage("include: alloc mem for eventhandler data\n");
   SCIP_CALL( SCIPallocBlockMemory(scip, &eventhdlrdata) );
   assert(eventhdlrdata != NULL);
   eventhdlrdata->gcg = gcg;
   SCIPdebugMessage("include: create hashmaps and set nsepas to 0\n");
   SCIP_CALL( SCIPhashmapCreate(&(eventhdlrdata->rowxgeneratedmap), SCIPblkmem(scip), STARTMAXCUTS) );

   eventhdlr = NULL;
   SCIP_CALL( SCIPincludeEventhdlrBasic(scip, &eventhdlr, EVENTHDLR_NAME, EVENTHDLR_DESC, eventExecEvent, eventhdlrdata) );
   assert(eventhdlr != NULL);
   gcg->mastersepacuthdlr = eventhdlr;

   /* set non fundamental callbacks via setter functions */
   SCIP_CALL( SCIPsetEventhdlrInit(scip, eventhdlr, eventInitMastersepacutUpdate) );
   SCIP_CALL( SCIPsetEventhdlrExitsol(scip, eventhdlr, eventExitSolMastersepacutUpdate) );
   SCIP_CALL( SCIPsetEventhdlrExit(scip, eventhdlr, eventExitMastersepacutUpdate) );
   SCIP_CALL( SCIPsetEventhdlrFree(scip, eventhdlr, eventFreeMastersepacutUpdate) );

   return SCIP_OKAY;
}


/** adds new mastersepacut to generated cuts */
SCIP_RETCODE GCGaddGeneratedMastersepacut(
   GCG*                          gcg,              /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut     /**< mastersepacut */
   )
{
   SCIP_EVENTHDLR*      eventhdlr;
   SCIP_EVENTHDLRDATA*  eventhdlrdata;
   SCIP_ROW*            row;

   assert(mastersepacut != NULL);
   assert(gcg != NULL);

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   row = GCGextendedmasterconsGetRow(mastersepacut);
   assert(row != NULL);

   SCIP_CALL( ensureGeneratedSize(GCGgetMasterprob(gcg), eventhdlrdata, eventhdlrdata->ngeneratedcuts + 1) );
   eventhdlrdata->generatedcuts[eventhdlrdata->ngeneratedcuts] = mastersepacut;
   SCIP_CALL( SCIPhashmapSetImageInt(eventhdlrdata->rowxgeneratedmap, row, eventhdlrdata->ngeneratedcuts) );
   (eventhdlrdata->ngeneratedcuts)++;

   SCIP_CALL( GCGcaptureMastersepacut(mastersepacut) );

   return SCIP_OKAY;
}


/** remove all mastersepacuts from generated cuts and release them */
SCIP_RETCODE GCGclearGeneratedMastersepacuts(
   GCG*              gcg                  /**< GCG data structure */
   )
{
   SCIP_EVENTHDLR* eventhdlr;
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   int j;

   assert(gcg != NULL);

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   /* global cuts were re-inserted to generated cuts: we can only clear after the separation round had been applied */
   if( eventhdlrdata->globalcutsreinserted )
   {
      eventhdlrdata->globalcutsreinserted = FALSE;
      return SCIP_OKAY;
   }

   SCIPdebugMessage("clear %i generated cuts \n", eventhdlrdata->ngeneratedcuts);
   for( j = 0; j < eventhdlrdata->ngeneratedcuts; j++ )
   {
      SCIP_CALL( GCGreleaseMastersepacut(gcg, &(eventhdlrdata->generatedcuts[j])) );
   }
   eventhdlrdata->ngeneratedcuts = 0;

   /* clear corresponding map */
   SCIPhashmapRemoveAll(eventhdlrdata->rowxgeneratedmap);

   return SCIP_OKAY;
}

/** removes the mastersepacuts which were applied at current node, but have already been removed from the LP */
SCIP_RETCODE GCGeventmastersepacutRemoveNewInactiveMastersepacuts(
   GCG*              gcg,           /**< GCG data structure */
   SCIP_EVENTHDLR*   eventhdlr,     /**< mastersepacut event handler */
   int               startidx       /**< indicate the first new mastersepacut */
   )
{
   GCG_EXTENDEDMASTERCONSDATA* mastercutdata;
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   SCIP_ROW* row;
   int j;

   assert(gcg != NULL);
   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   for( j = eventhdlrdata->nactivecuts - 1; j >= startidx; --j )
   {
      mastercutdata = eventhdlrdata->activecuts[j];
      assert(mastercutdata != NULL);
      row = GCGextendedmasterconsGetRow(mastercutdata);
      assert(row != NULL);

      /* rows still in LP remain relevant and therefore stay in active cuts */
      if( SCIProwIsInLP(row) )
         continue;

      /* cut was created and removed at same node: can be removed from active cuts and released */
      SCIPdebugMessage("remove new inactive rows: remove row %s and free data\n", SCIProwGetName(row));
      SCIP_CALL( GCGreleaseMastersepacut(gcg, &(eventhdlrdata->activecuts[j])) );
      assert(eventhdlrdata->activecuts[j] == NULL);
      (eventhdlrdata->nactivecuts)--;
      eventhdlrdata->activecuts[j] = eventhdlrdata->activecuts[eventhdlrdata->nactivecuts];
   }

   return SCIP_OKAY;
}

/** removes the mastersepacuts which were applied at current node, but have already been removed from the LP */
SCIP_RETCODE GCGremoveNewInactiveMastersepacuts(
   GCG*              gcg,           /**< GCG data structure */
   int               startidx       /**< index of the first new separaot mastercut */
   )
{
   SCIP_EVENTHDLR* eventhdlr;

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   SCIP_CALL( GCGeventmastersepacutRemoveNewInactiveMastersepacuts(gcg, eventhdlr, startidx) );

   return SCIP_OKAY;
}


/** removes all mastersepacuts after given index from active cuts */
SCIP_RETCODE GCGeventmastersepacutShrinkActiveMastersepacuts(
   GCG*              gcg,             /**< GCG data structure */
   SCIP_EVENTHDLR*   eventhdlr,       /**< mastersepacuts event handler */
   int               newnrows         /**< index to which active cuts should be shrunk to */
   )
{
   SCIP* masterscip;
   GCG_EXTENDEDMASTERCONSDATA* mastercutdata;
   SCIP_EVENTHDLRDATA* eventhdlrdata;
   int j;

   assert(gcg != NULL);
   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   masterscip = GCGgetMasterprob(gcg);

   /* master problems is already beyond SCIP_STAGE_EXITSOLVE --> activecuts has already been cleared */
   if( SCIPgetStage(masterscip) > SCIP_STAGE_EXITSOLVE )
      return SCIP_OKAY;

   if( eventhdlrdata->nactivecuts == newnrows )
      return SCIP_OKAY;

   assert(eventhdlrdata->nactivecuts >= newnrows);

   /* release all cuts added to active cuts after newnrows */
   for( j = eventhdlrdata->nactivecuts - 1; j >= newnrows; --j )
   {
      SCIP_ROW* row;

      mastercutdata = eventhdlrdata->activecuts[j];
      assert(mastercutdata != NULL);
      row = GCGextendedmasterconsGetRow(mastercutdata);
      assert(row != NULL);

      /* global rows with age zero will be added to the separation storage before initial LP is constructed
       * --> these rows will be re-added to the next LP
       * --> corresponding mastersepacut is added back to generated cuts */
      if( masterscip->tree->correctlpdepth != -1 && !SCIProwIsLocal(row) && SCIProwGetAge(row) == 0 )
      {
         SCIP_CALL( reinsertGlobalMastersepacut(gcg, eventhdlrdata->activecuts[j], eventhdlrdata) );
         eventhdlrdata->globalcutsreinserted = TRUE;
      }

      SCIP_CALL( GCGreleaseMastersepacut(gcg, &(eventhdlrdata->activecuts[j])) );
   }
   eventhdlrdata->nactivecuts = newnrows;

   return SCIP_OKAY;
}

/** removes all mastersepacuts after given index from active cuts */
SCIP_RETCODE GCGshrinkActiveMastersepacuts(
   GCG*              gcg,             /**< GCG data structure */
   int               newnrows         /**< index to which active cuts should be shrunk to */
   )
{
   SCIP_EVENTHDLR* eventhdlr;

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   SCIP_CALL( GCGeventmastersepacutShrinkActiveMastersepacuts(gcg, eventhdlr, newnrows) );

   return SCIP_OKAY;
}

/** returns active master separator cuts */
GCG_EXTENDEDMASTERCONSDATA** GCGeventmastersepacutGetActiveMastersepacuts(
   GCG*              gcg,            /**< GCG data structure */
   SCIP_EVENTHDLR*   eventhdlr       /**< mastersepacut event handler*/
   )
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   return eventhdlrdata->activecuts;
}

/** returns active master separator cuts */
GCG_EXTENDEDMASTERCONSDATA** GCGgetActiveMastersepacuts(
   GCG*              gcg            /**< GCG data structure */
   )
{
   SCIP_EVENTHDLR* eventhdlr;

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   return GCGeventmastersepacutGetActiveMastersepacuts(gcg, eventhdlr);
}

/** return number of active master separator cuts */
int GCGeventmastersepacutGetNActiveMastersepacuts(
   GCG*              gcg,            /**< GCG data structure */
   SCIP_EVENTHDLR*   eventhdlr       /**< masterepacut event handler */
   )
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   return eventhdlrdata->nactivecuts;
}

/** return number of active mastersepacuts */
int GCGgetNActiveMastersepacuts(
   GCG*              gcg            /**< GCG data structure */
   )
{
   SCIP_EVENTHDLR* eventhdlr;

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   return GCGeventmastersepacutGetNActiveMastersepacuts(gcg, eventhdlr);
}


/** adds a mastercut generated by a master separator to active cuts */
SCIP_RETCODE GCGeventmastersepacutAddActiveMastersepacut(
   GCG*                          gcg,              /**< GCG data structure */
   SCIP_EVENTHDLR*               eventhdlr,        /**< mastersepacut event handler */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut     /**< mastersepacut */
   )
{
   SCIP_EVENTHDLRDATA* eventhdlrdata;

   assert(eventhdlr != NULL);

   eventhdlrdata = SCIPeventhdlrGetData(eventhdlr);
   assert(eventhdlrdata != NULL);

   addToActiveMastersepacuts(gcg, mastersepacut, eventhdlrdata);

   return SCIP_OKAY;
}

/** adds a mastercut generated by a master separator to active cuts */
SCIP_RETCODE GCGaddActiveMastersepacut(
   GCG*                          gcg,              /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut     /**< mastersepacut */
   )
{
   SCIP_EVENTHDLR* eventhdlr;

   eventhdlr = GCGgetMastersepacutEventhdlr(gcg);
   assert(eventhdlr != NULL);

   SCIP_CALL( GCGeventmastersepacutAddActiveMastersepacut(gcg, eventhdlr, mastersepacut) );

   return SCIP_OKAY;
}
