/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*                  This file is part of the program and library             */
/*          GCG --- Generic Column Generation                                */
/*                  a Dantzig-Wolfe decomposition based extension            */
/*                  of the branch-cut-and-price framework                    */
/*         SCIP --- Solving Constraint Integer Programs                      */
/*                                                                           */
/* Copyright (C) 2010-2026 Operations Research, RWTH Aachen University       */
/*                         Zuse Institute Berlin (ZIB)                       */
/*                                                                           */
/*  Licensed under the Apache License, Version 2.0 (the "License");          */
/*  you may not use this file except in compliance with the License.         */
/*  You may obtain a copy of the License at                                  */
/*                                                                           */
/*      http://www.apache.org/licenses/LICENSE-2.0                           */
/*                                                                           */
/*  Unless required by applicable law or agreed to in writing, software      */
/*  distributed under the License is distributed on an "AS IS" BASIS,        */
/*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. */
/*  See the License for the specific language governing permissions and      */
/*  limitations under the License.                                           */
/*                                                                           */
/*  You should have received a copy of the Apache-2.0 license                */
/*  along with GCG; see the file LICENSE. If not visit gcg.or.rwth-aachen.de.*/
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file   mastersepacut.c
 * @brief  data structures to store all the relevant data for cuts generated by master separators
 * @author Chantal Reinartz Groba
 * @author Erik Muehmer
 */

/*---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/
#include <scip/type_scip.h>

#include "gcg/gcg.h"
#include "gcg/gcgvarhistory.h"
#include "gcg/mastersepacut.h"
#include "gcg/pricer_gcg.h"
#include "gcg/struct_extendedmasterconsdata.h"
#include "gcg/struct_mastersepacut.h"
#include "gcg/struct_sepagcg.h"
#include "gcg/struct_gcgcol.h"


/** frees separator master cut */
static
SCIP_RETCODE freeMastersepacut(
   GCG*                          gcg,              /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*   mastersepacut     /**< pointer to separator master cut */
   )
{
   SCIP* masterscip;
   GCG_MASTERSEPACUT* cut;
   GCG_SEPA* sepa;
   assert(gcg != NULL);
   assert(mastersepacut != NULL);

   masterscip = GCGgetMasterprob(gcg);
   cut = mastersepacut->data.mastersepacut;
   sepa = cut->sepa;

   if( cut->knownvarhistory != NULL )
   {
      SCIPdebugMessage("free mastersepacut: var history is freed\n");
      SCIP_CALL( GCGvarhistoryFreeReference(masterscip, &(cut->knownvarhistory)) );
   }
   assert(cut->knownvarhistory == NULL);

   assert(cut->data == NULL || sepa->sepamastercutdelete != NULL);
   if(cut->data != NULL && sepa->sepamastercutdelete != NULL )
   {
      SCIP_CALL( (*sepa->sepamastercutdelete)(gcg, sepa, mastersepacut, &cut->data) );
   }
   assert(cut->data == NULL);

   SCIPfreeBlockMemory(masterscip, &mastersepacut->data.mastersepacut);
   mastersepacut->data.mastersepacut = NULL;

   return SCIP_OKAY;
}

/** increases usage counter of separator master cut */
SCIP_RETCODE GCGcaptureMastersepacut(
   GCG_EXTENDEDMASTERCONSDATA*      mastersepacut      /**< separator master cut */
   )
{
   GCG_MASTERSEPACUT* cut;
   assert(mastersepacut != NULL);
   assert(GCGextendedmasterconsGetType(mastersepacut) == GCG_EXTENDEDMASTERCONSTYPE_SEPA_ROW);

   cut = mastersepacut->data.mastersepacut;
   assert(cut->nuses >= 0);
   cut->nuses++;

   return SCIP_OKAY;
}

/** decreases usage counter of separator master cut, and frees memory if necessary */
SCIP_RETCODE GCGreleaseMastersepacut(
   GCG*                             gcg,             /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA**     mastersepacut    /**< pointer to separator master cut */
   )
{
   GCG_MASTERSEPACUT* cut;
   assert(gcg != NULL);
   assert(mastersepacut != NULL);
   assert(*mastersepacut != NULL);
   assert(GCGextendedmasterconsGetType(*mastersepacut) == GCG_EXTENDEDMASTERCONSTYPE_SEPA_ROW);

   cut = (*mastersepacut)->data.mastersepacut;
   assert(cut->nuses > 0);

   cut->nuses--;
   if( cut->nuses == 0 )
   {
      SCIP_CALL( freeMastersepacut(gcg, *mastersepacut) );
      SCIP_CALL( GCGextendedmasterconsFree(gcg, mastersepacut) );
   }

   *mastersepacut = NULL;

   return SCIP_OKAY;
}

/**< creates separator master cut */
SCIP_RETCODE GCGcreateMastersepacut(
   GCG*                         gcg,                 /**< GCG data structure */
   GCG_MASTERSEPACUT**     mastersepacut,       /**< pointer to store separator master cut */
   GCG_SEPA*                    sepa,                /**< separator creating this cut */
   GCG_VARHISTORY*              varhistory,          /**< variable history */
   GCG_MASTERSEPACUTDATA*  mastersepacutdata    /**< separator master cut data */
   )
{
   assert(gcg != NULL);
   assert(mastersepacut != NULL);

   SCIP_CALL( SCIPallocBlockMemory(GCGgetMasterprob(gcg), mastersepacut) );
   (*mastersepacut)->nuses = 0;
   (*mastersepacut)->knownvarhistory = varhistory;
   (*mastersepacut)->data = mastersepacutdata;
   (*mastersepacut)->sepa = sepa;

   return SCIP_OKAY;
}

#ifndef NDEBUG
/**< returns the variable history of the separator master cut */
GCG_VARHISTORY* GCGmastersepacutGetVarHistory(
   GCG_MASTERSEPACUT*      mastersepacut     /**< separator master cut */
   )
{
   assert(mastersepacut != NULL);

   return mastersepacut->knownvarhistory;
}
#endif

/**< return the separator which created the cut */
GCG_SEPA* GCGmastersepacutGetSeparator(
   GCG_MASTERSEPACUT*      mastersepacut     /**< separator master cut */
   )
{
   assert(mastersepacut != NULL);

   return mastersepacut->sepa;
}

/**< returns the data of the separator master cut */
GCG_MASTERSEPACUTDATA* GCGmastersepacutGetData(
   GCG_MASTERSEPACUT*      mastersepacut     /**< separator master cut */
   )
{
   assert(mastersepacut != NULL);

   return mastersepacut->data;
}

/**< set the variable history of separator master cut */
SCIP_RETCODE GCGmastersepacutSetVarHistory(
   GCG*                         gcg,              /**< GCG data structure */
   GCG_MASTERSEPACUT*      mastersepacut     /**< pointer to separator master cut */
   )
{
   SCIP* masterscip;
   assert(gcg != NULL);
   assert(mastersepacut != NULL);

   masterscip = GCGgetMasterprob(gcg);
   SCIP_CALL( GCGvarhistoryCopyReference(masterscip, &(mastersepacut->knownvarhistory), GCGgetCurrentVarhistoryReference(gcg)) );
   return SCIP_OKAY;
}

#ifndef NDEBUG
/** returns the corresponding mastersepacut */
GCG_MASTERSEPACUT* GCGextendedmasterconsGetMastersepacut(
   GCG_EXTENDEDMASTERCONSDATA* extendedmasterconsdata
   )
{
   assert(extendedmasterconsdata != NULL);
   assert(GCGextendedmasterconsGetType(extendedmasterconsdata) == GCG_EXTENDEDMASTERCONSTYPE_SEPA_ROW);
   return extendedmasterconsdata->data.mastersepacut;
}
#endif

/** creates extended master cons data for a given master cut */
SCIP_RETCODE GCGextendedmasterconsCreateForSepamastercut(
   GCG*                          gcg,                       /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA**  extendedmastercons,        /**< pointer to store the create extended master cons data */
   GCG_MASTERSEPACUT*       cut,                       /**< master cut data structure */
   SCIP_ROW*                     row,                       /**< corresponding row object */
   GCG_PRICINGMODIFICATION**     pricingmodifications,      /**< related pricing modifications */
   int                           npricingmodifications      /**< number of pricing modifications */
   )
{
   assert(cut != NULL);
   assert(row != NULL);
   assert(pricingmodifications != NULL || npricingmodifications == 0);
   assert(extendedmastercons != NULL);
   SCIP_CALL( GCGextendedmasterconsCreateFromRow(gcg, extendedmastercons, GCG_EXTENDEDMASTERCONSTYPE_SEPA_ROW, row, pricingmodifications, npricingmodifications, cut) );
   return SCIP_OKAY;
}

SCIP_RETCODE GCGmastersepacutGetExtendedmasterconsCoeff(
   GCG*                             gcg,                          /**< GCG data structure */
   GCG_EXTENDEDMASTERCONSDATA*      mastersepacut,                /**< master cut data structure */
   SCIP_VAR**                       solvars,                      /**< array of column solution variables */
   SCIP_Real*                       solvals,                      /**< array of column solution values */
   int                              nsolvars,                     /**< number of column solution variables and values */
   int                              probnr,                       /**< the pricing problem that the column belongs to */
   GCG_COL*                         gcgcol,                       /**< gcg column if available (or NULL) */
   SCIP_Real*                       coef                          /**< pointer to store the coefficient */
   )
{
   GCG_SEPA* sepa = GCGmastersepacutGetSeparator(GCGextendedmasterconsGetMastersepacut(mastersepacut));

   SCIP_CALL( sepa->sepagetcolcoefficient(gcg, sepa, mastersepacut, solvars, solvals, nsolvars, probnr, gcgcol, coef) );
   return SCIP_OKAY;
}
