/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*                  This file is part of the program and library             */
/*          GCG --- Generic Column Generation                                */
/*                  a Dantzig-Wolfe decomposition based extension            */
/*                  of the branch-cut-and-price framework                    */
/*         SCIP --- Solving Constraint Integer Programs                      */
/*                                                                           */
/* Copyright (C) 2010-2026 Operations Research, RWTH Aachen University       */
/*                         Zuse Institute Berlin (ZIB)                       */
/*                                                                           */
/*  Licensed under the Apache License, Version 2.0 (the "License");          */
/*  you may not use this file except in compliance with the License.         */
/*  You may obtain a copy of the License at                                  */
/*                                                                           */
/*      http://www.apache.org/licenses/LICENSE-2.0                           */
/*                                                                           */
/*  Unless required by applicable law or agreed to in writing, software      */
/*  distributed under the License is distributed on an "AS IS" BASIS,        */
/*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. */
/*  See the License for the specific language governing permissions and      */
/*  limitations under the License.                                           */
/*                                                                           */
/*  You should have received a copy of the Apache-2.0 license                */
/*  along with GCG; see the file LICENSE. If not visit gcg.or.rwth-aachen.de.*/
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file   pub_gcgvar.h
 * @ingroup PUBLICCOREAPI
 * @brief  public methods for GCG variables
 * @author Martin Bergner
 * @author Christian Puchert
 */

/*---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/

#ifndef GCG_PUB_GCGVAR_H__
#define GCG_PUB_GCGVAR_H__

#include "gcg/type_decomp.h"
#include "scip/scip.h"

#ifdef NDEBUG
#include "gcg/struct_vardata.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @ingroup TYPEDEFINITIONS
 *
 * @{
 */

#ifdef NDEBUG
#define GCGvarIsPricing(var)                (SCIPvarGetData(var)->vartype == GCG_VARTYPE_PRICING)
#else
/** returns TRUE or FALSE whether variable is a pricing variable or not */
GCG_EXPORT
SCIP_Bool GCGvarIsPricing(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGvarIsOriginal(var)                (SCIPvarGetData(var)->vartype == GCG_VARTYPE_ORIGINAL)
#else
/** returns TRUE or FALSE whether variable is a original variable or not */
GCG_EXPORT
SCIP_Bool GCGvarIsOriginal(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGvarIsMaster(var)                  (SCIPvarGetData(var)->vartype == GCG_VARTYPE_MASTER)
#else
/** returns TRUE or FALSE whether variable is a master variable or not */
GCG_EXPORT
SCIP_Bool GCGvarIsMaster(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGvarIsInferredPricing(var)         (SCIPvarGetData(var)->vartype == GCG_VARTYPE_INFERREDPRICING)
#else
/** returns TRUE or FALSE whether variable is a inferred pricing variable or not
  *
  * inferred pricing variables are auxilary variables that are required by specific extended master conss */
GCG_EXPORT
SCIP_Bool GCGvarIsInferredPricing(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

/** count the number of inferred (coefficient) pricing variables in a array of variables */
int GCGcountInferredCoefPricingVars(
   SCIP_VAR**             vars,               /**< array of variables */
   int                    nvars               /**< number of variables */
   );

#ifdef NDEBUG
#define GCGoriginalVarIsLinking(var)         (SCIPvarGetData(var)->blocknr == -2)
#else
/** returns TRUE or FALSE whether variable is a linking variable or not */
GCG_EXPORT
SCIP_Bool GCGoriginalVarIsLinking(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarIsTransVar(var)        (SCIPvarGetData(var)->blocknr == -1)
#else
/** returns TRUE or FALSE whether variable is a directly transferred variable or not */
GCG_EXPORT
SCIP_Bool GCGoriginalVarIsTransVar(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGpricingVarGetOriginalVar(var)     (SCIPvarGetData(var)->data.pricingvardata.origvars[0])
#else
/** returns the original var of a pricing variable */
GCG_EXPORT
SCIP_VAR* GCGpricingVarGetOriginalVar(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

/** adds the original var to the pricing variable */
GCG_EXPORT
SCIP_RETCODE GCGpricingVarAddOrigVar(
   SCIP*                 pricingprob,        /**< pricingprob SCIP data structure */
   SCIP_VAR*             pricingvar,         /**< pricing variable */
   SCIP_VAR*             origvar             /**< original pricing variable */
   );

#ifdef NDEBUG
#define GCGoriginalVarGetPricingVar(var)     (SCIPvarGetData(var)->data.origvardata.pricingvar)
#else
/** returns the pricing var of an original variable */
GCG_EXPORT
SCIP_VAR* GCGoriginalVarGetPricingVar(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** sets the pricing var of an original variable */
GCG_EXPORT
void GCGoriginalVarSetPricingVar(
   SCIP_VAR*             var,                /**< SCIP variable structure */
   SCIP_VAR*             pricingvar          /**< SCIP variable structure */
   );

/** copies the pricing variable data to a master problem variable. This is used in the Benders' decomposition mode when
 * subproblems are merged into the master problem.
 */
GCG_EXPORT
SCIP_RETCODE GCGcopyPricingvarDataToMastervar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             pricingvar,         /**< the pricing problem variable is copied from */
   SCIP_VAR*             mastervar           /**< the master variable that the vardata is copied to */
   );

#ifdef NDEBUG
#define GCGlinkingVarGetPricingVars(var)     (SCIPvarGetData(var)->data.origvardata.linkingvardata->pricingvars)
#else
/** returns the pricing variables of an linking variable */
GCG_EXPORT
SCIP_VAR** GCGlinkingVarGetPricingVars(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

/** sets the pricing var of the corresponding linking variable at the specified position */
GCG_EXPORT
void GCGlinkingVarSetPricingVar(
   SCIP_VAR*             origvar,            /**< original variable */
   int                   pricingprobnr,      /**< number of pricing problem */
   SCIP_VAR*             var                 /**< pricing variable */
   );

#ifdef NDEBUG
#define GCGoriginalVarGetNMastervars(var)    (SCIPvarGetData(var)->data.origvardata.nmastervars)
#else
/** returns the number of master variables the original variable is contained in */
GCG_EXPORT
int GCGoriginalVarGetNMastervars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarGetMastervars(var)     (SCIPvarGetData(var)->data.origvardata.mastervars)
#else
/** returns the master variables the original variable is contained in */
GCG_EXPORT
SCIP_VAR** GCGoriginalVarGetMastervars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarGetMastervals(var)     (SCIPvarGetData(var)->data.origvardata.mastervals)
#else
/** returns the fraction of master variables the original variable is contained in */
GCG_EXPORT
SCIP_Real* GCGoriginalVarGetMastervals(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarGetCoefs(var)          (SCIPvarGetData(var)->data.origvardata.coefs)
#else
/** returns the fraction of master variables the original variable is contained in */
GCG_EXPORT
SCIP_Real* GCGoriginalVarGetCoefs(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarGetMasterconss(var)    (SCIPvarGetData(var)->data.origvardata.masterconss)
#else
/** returns the fraction of master variables the original variable is contained in */
GCG_EXPORT
SCIP_CONS** GCGoriginalVarGetMasterconss(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** adds a coefficient of the master variable to the coefs array for the resp. constraint */
GCG_EXPORT
SCIP_RETCODE GCGoriginalVarAddCoef(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             var,                /**< variable to add coef */
   SCIP_Real             val,                /**< coefficent to set */
   SCIP_CONS*            cons                /**< constraint the variable is in */
   );

/** adds variable to a new block, making a linkingvariable out of it, if necessary */
GCG_EXPORT
SCIP_RETCODE GCGoriginalVarAddBlock(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             var,                /**< var that is added to a block */
   int                   newblock,           /**< the new block the variable will be in */
   int                   nblocks,            /**< total number of pricing problems */
   GCG_DECMODE           mode                /**< the decomposition mode */
   );

#ifdef NDEBUG
#define GCGlinkingVarGetLinkingConss(var)    (SCIPvarGetData(var)->data.origvardata.linkingvardata->linkconss)
#else
/** returns the linking constraints */
GCG_EXPORT
SCIP_CONS** GCGlinkingVarGetLinkingConss(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** sets the linking constraints*/
GCG_EXPORT
void GCGlinkingVarSetLinkingCons(
   SCIP_VAR*             var,                /**< variable data structure */
   SCIP_CONS*            cons,               /**< linking constraint */
   int                   index               /**< index of pricing problem */
   );

/** returns the blocks the linking variable is in */
GCG_EXPORT
SCIP_RETCODE GCGlinkingVarGetBlocks(
   SCIP_VAR*             var,                /**< SCIP variable structure */
   int                   nblocks,            /**< number of blocks the linking variable is in */
   int*                  blocks              /**< array to store the blocks of the linking variable */
   );

#ifdef NDEBUG
#define GCGlinkingVarGetNBlocks(var)         (SCIPvarGetData(var)->data.origvardata.linkingvardata->nblocks)
#else
/** returns the number of blocks the linking variable is in */
GCG_EXPORT
int GCGlinkingVarGetNBlocks(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGoriginalVarGetNCoefs(var)         (SCIPvarGetData(var)->data.origvardata.ncoefs)
#else
/** returns the number of coefficients of master constraints the original variable is contained in */
GCG_EXPORT
int GCGoriginalVarGetNCoefs(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

/** sets the number of master variables the original variable is contained in */
GCG_EXPORT
void GCGoriginalVarSetNCoefs(
   SCIP_VAR*             var,                /**< SCIP variable structure */
   int                   coef                /**< number of coefficient to set */
   );

/** returns TRUE or FALSE whether a master variable is a direct copy of a linking variable or not */
GCG_EXPORT
SCIP_Bool GCGmasterVarIsLinking(
   SCIP_VAR*             var                 /**< variable data structure */
   );

/** returns scip instance corresponding to master variable */
GCG_EXPORT
SCIP* GCGmasterVarGetProb(
   SCIP_VAR*             var                 /**< variable data structure */
   );

#ifdef NDEBUG
#define GCGmasterVarIsRay(var)               (SCIPvarGetData(var)->data.mastervardata.isray)
#else
/** returns whether the master variable is a ray */
GCG_EXPORT
SCIP_Bool GCGmasterVarIsRay(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGmasterVarIsArtificial(var)        (SCIPvarGetData(var)->data.mastervardata.isartificial)
#else
/** returns TRUE or FALSE whether a master variable is an artificial variable */
GCG_EXPORT
SCIP_Bool GCGmasterVarIsArtificial(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGmasterVarGetNOrigvars(var)        (SCIPvarGetData(var)->data.mastervardata.norigvars)
#else
/** returns the number of original variables the master variable is contained in */
GCG_EXPORT
int GCGmasterVarGetNOrigvars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGmasterVarGetOrigvars(var)         (SCIPvarGetData(var)->data.mastervardata.origvars)
#else
/** returns the original variables the master variable is contained in */
GCG_EXPORT
SCIP_VAR** GCGmasterVarGetOrigvars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGmasterVarGetOrigvals(var)         (SCIPvarGetData(var)->data.mastervardata.origvals)
#else
/** returns the fraction of original variables the master variable is contained in */
GCG_EXPORT
SCIP_Real* GCGmasterVarGetOrigvals(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** returns the fraction of the original variable the master variable is contained in */
GCG_EXPORT
SCIP_Real GCGmasterVarGetOrigval(
   SCIP_VAR*             mastervar,         /**< master variable data structure */
   SCIP_VAR*             origvar            /**< original variable data structure */
   );

#ifdef NDEBUG
#define GCGmasterVarGetOrigvalmap(var)       (SCIPvarGetData(var)->data.mastervardata.origvar2val)
#else
/** returns a hash map that stores the fraction of original variables the master variable is contained in */
GCG_EXPORT
SCIP_HASHMAP* GCGmasterVarGetOrigvalmap(
   SCIP_VAR*             var              /**< master variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGpricingVarGetNOrigvars(var)       (SCIPvarGetData(var)->data.pricingvardata.norigvars)
#else
/** returns the number of original variables the pricing variable is contained in */
GCG_EXPORT
int GCGpricingVarGetNOrigvars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGpricingVarGetOrigvars(var)        (SCIPvarGetData(var)->data.pricingvardata.origvars)
#else
/** returns the original variables the pricing variable is contained in */
GCG_EXPORT
SCIP_VAR** GCGpricingVarGetOrigvars(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGvarGetBlock(var)             (SCIPvarGetData(var)->blocknr)
#else
/** returns the block of the variable */
GCG_EXPORT
int GCGvarGetBlock(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** sets the block of the variable */
GCG_EXPORT
void GCGvarSetBlock(
   SCIP_VAR*             var,                /**< variable to set block for */
   int                   block               /**< block to set */
   );

/** creates the data for all variables of the original program */
GCG_EXPORT
SCIP_RETCODE GCGcreateOrigVarsData(
   GCG*                  gcg                 /**< GCG data structure */
   );

/** frees the data for all variables of the original program */
GCG_EXPORT
SCIP_RETCODE GCGfreeOrigVarsData(
   GCG*                  gcg                 /**< GCG data structure */
   );

/** creates the data for a variable of the original program */
GCG_EXPORT
SCIP_RETCODE GCGorigVarCreateData(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             var                 /**< pointer to variable object */
   );

#ifdef NDEBUG
#define GCGisLinkingVarInBlock(var, block)   (GCGlinkingVarGetPricingVars(var)[block] != NULL)
#else
/** returns TRUE if the linking variable is in the block, FALSE otherwise */
GCG_EXPORT
SCIP_Bool GCGisLinkingVarInBlock(
   SCIP_VAR*             var,                /**< variable data structure */
   int                   block               /**< pricing problem number */
   );
#endif

/** determines if the master variable is in the given block */
GCG_EXPORT
SCIP_Bool GCGisMasterVarInBlock(
   SCIP_VAR*             mastervar,          /**< master variable */
   int                   blocknr             /**< block number to check */
   );

/** informs an original variable, that a variable in the master problem was created,
 * that contains a part of the original variable.
 * Saves this information in the original variable's data
 */
GCG_EXPORT
SCIP_RETCODE GCGoriginalVarAddMasterVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             origvar,            /**< original variable */
   SCIP_VAR*             var,                /**< master variable */
   SCIP_Real             val                 /**< fraction of the original variable */
   );

/* informs an original variable, that a variable in the master problem was deleted,
 * that contains a part of the original variable.
 * Update the information in the original variable's data
 */
GCG_EXPORT
SCIP_RETCODE GCGoriginalVarRemoveMasterVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             origvar,            /**< original variable */
   SCIP_VAR*             var                 /**< master variable */
   );


/** creates the corresponding pricing variable for the given original variable */
GCG_EXPORT
SCIP_RETCODE GCGoriginalVarCreatePricingVar(
   SCIP*                 pricingprob,        /**< pricingprob SCIP data structure */
   SCIP_VAR*             origvar,            /**< original variable */
   SCIP_VAR**            var                 /**< pricing variable */
   );

/** creates the corresponding pricing variable for the given original variable */
GCG_EXPORT
SCIP_RETCODE GCGlinkingVarCreatePricingVar(
   SCIP*                 pricingscip,        /**< pricing problem SCIP data structure */
   int                   pricingprobnr,      /**< number of the pricing problem */
   SCIP_VAR*             origvar,            /**< original variable */
   SCIP_VAR**            var                 /**< pointer to store new pricing variable */
   );

/** creates the corresponding constraint in the master problem for the linking variable */
GCG_EXPORT
SCIP_RETCODE GCGlinkingVarCreateMasterCons(
   GCG*                  gcg,                /**< GCG data structure */
   int                   pricingprobnr,      /**< number of the pricing problem */
   SCIP_VAR*             origvar,            /**< original variable */
   SCIP_CONS**           linkcons            /**< constraint linking pricing variables */
   );

/** creates the master var and initializes the vardata */
GCG_EXPORT
SCIP_RETCODE GCGcreateMasterVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP*                 pricingscip,        /**< pricing problem SCIP data structure */
   SCIP_VAR**            newvar,             /**< pointer to store new master variable */
   const char*           varname,            /**< new variable name */
   SCIP_Real             objcoeff,           /**< new objective coeffient */
   SCIP_VARTYPE          vartype,            /**< new variable type */
   SCIP_Bool             solisray,           /**< indicates whether new variable is a ray */
   int                   prob,               /**< number of pricing problem that created this variable */
   int                   nsolvars,           /**< number of variables in the solution */
   SCIP_Real*            solvals,            /**< values of variables in the solution */
   SCIP_VAR**            solvars,            /**< variables with non zero coefficient in the solution */
   SCIP_Bool             auxiliaryvar        /**< is new variable an Benders' auxiliary variables? */
   );

/** creates initial master variables and the vardata */
GCG_EXPORT
SCIP_RETCODE GCGcreateInitialMasterVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             var,                /**< original variable */
   SCIP_VAR**            newvar              /**< pointer to store new variable */
   );

/** creates artificial variable and the vardata */
GCG_EXPORT
SCIP_RETCODE GCGcreateArtificialVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR**            newvar,              /**< pointer to store new variable */
   const char*           name,               /**< name of variable, or NULL for automatic name creation */
   SCIP_Real             objcoef             /**< objective coefficient of artificial variable */
   );

/** creates a pricing variable inferred from an extended master cons
 * that does not correspond to any original variable and its vardata */
SCIP_RETCODE GCGcreateInferredPricingVar(
   SCIP*                 pricingscip,        /**< pricing problem SCIP data structure */
   SCIP_VAR**            newvar,             /**< pointer to store new master variable */
   const char*           varname,            /**< new variable name */
   SCIP_Real             lb,                 /**< new variable lower bound */
   SCIP_Real             ub,                 /**< new objective coefficient */
   SCIP_Bool             iscoefvar,          /**< is this a coefficient variable? (objcoeff can be 0 if TRUE but not != 0 if FALSE) */
   SCIP_Real             objcoeff,           /**< new objective coefficient */
   SCIP_VARTYPE          vartype,            /**< new variable type */
   int                   prob                /**< number of pricing problem that created this variable */
   );

/* adds the vardata to the auxiliary variable */
GCG_EXPORT
SCIP_RETCODE GCGaddDataAuxiliaryVar(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             auxiliaryvar,       /**< the auxiliary variable */
   int                   probnumber          /**< the subproblem number */
   );

/** sets the creation node of this var */
GCG_EXPORT
void GCGsetCreationNode(
   SCIP_VAR*             var,                /**< created variable */
   SCIP_Longint          creationNode        /**< node */
   );

#ifdef NDEBUG
#define GCGgetCreationNode(var)              (SCIPvarGetData(var)->creationnode)
#else
/** returns the creation node of this var */
GCG_EXPORT
long long int GCGgetCreationNode(
   SCIP_VAR*             var                 /**< created variable */
   );
#endif

/** sets the creation time of this var */
GCG_EXPORT
void GCGsetCreationTime(
   SCIP_VAR*             var,                /**< created variable */
   SCIP_Real             time                /**< creation time */
   );

#ifdef NDEBUG
#define GCGgetCreationTime(var)              (SCIPvarGetData(var)->creationtime)
#else
/** returns the creation time of this var */
GCG_EXPORT
SCIP_Real GCGgetCreationTime(
   SCIP_VAR*             var                 /**< created variable */
   );
#endif

/** store pricing reduced cost call */
GCG_EXPORT
void GCGsetRootRedcostCall(
   SCIP_VAR*             var,                /**< variable data structure */
   SCIP_Longint          rootredcostcall     /**< iteration at which the variable is created */
   );

#ifdef NDEBUG
#define GCGgetRootRedcostCall(var)           (SCIPvarGetData(var)->rootredcostcall)
#else
/** return stored pricing reduced cost call */
GCG_EXPORT
SCIP_Longint GCGgetRootRedcostCall(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

#ifdef NDEBUG
#define GCGgetIteration(var)                 (SCIPvarGetData(var)->iteration)
#else
/** returns the iteration when the var was created */
GCG_EXPORT
SCIP_Longint GCGgetIteration(
   SCIP_VAR*             var                 /**< created variable */
   );
#endif

/** sets the iteration when the var was created */
GCG_EXPORT
void GCGsetIteration(
   SCIP_VAR*             var,                /**< created variable */
   SCIP_Longint          iteration           /**< iteration that this var was created */
   );

/** store gap */
GCG_EXPORT
void GCGsetVarGap(
   SCIP_VAR*             var,                /**< variable data structure */
   SCIP_Real             gap                 /**< present gap when variable is created */
   );

#ifdef NDEBUG
#define GCGgetVarGap(var)                    (SCIPvarGetData(var)->gap)
#else
/** return stored gap */
GCG_EXPORT
SCIP_Real GCGgetVarGap(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** store reduced cost */
GCG_EXPORT
void GCGsetRedcost(
   GCG*                  gcg,                /**< GCG data structure */
   SCIP_VAR*             var,                /**< variable data structure */
   SCIP_Real             redcost             /**< reduced cost of the variable at creation */
   );

#ifdef NDEBUG
#define GCGgetRedcost(var)                   (SCIPvarGetData(var)->redcost)
#else
/** return stored reduced cost */
GCG_EXPORT
SCIP_Real GCGgetRedcost(
   SCIP_VAR*             var                 /**< variable data structure */
   );
#endif

/** updates the statistics part of the variable */
GCG_EXPORT
void GCGupdateVarStatistics(
    GCG*                 gcg,                /**< GCG data structure */
    SCIP_VAR*            newvar,             /**< new variable for statistic update */
    SCIP_Real            redcost             /**< reduced cost of the variable */
   );

/** prints the given variable: name, type (original, master or pricing) block number,
 * and the list of all variables related to the given variable */
GCG_EXPORT
void GCGprintVar(
   GCG*                  gcg,                /**< GCG data structure */
   FILE*                 file,               /**< File to write information to, or NULL for stdout */
   SCIP_VAR*             var                 /**< variable that should be printed */
   );

#ifdef NDEBUG
#define GCGmasterVarGetIndex(var)            (SCIPvarGetData(var)->data.mastervardata.index)
#else
/** returns the index of the mater variable */
GCG_EXPORT
int GCGmasterVarGetIndex(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

/** sets the index of the master variable */
GCG_EXPORT
void GCGmasterVarSetIndex(
   SCIP_VAR*             var,                /**< SCIP variable structure */
   int                   index               /**< index */
   );

#ifdef NDEBUG
#define GCGinferredPricingVarGetExtendedmasterconsdata(var)            (SCIPvarGetData(var)->data.inferredpricingvardata.extendedmasterconsdata)
#else
/* returns the extended master cons data of the inferred pricing var */
GCG_EXPORT
GCG_EXTENDEDMASTERCONSDATA* GCGinferredPricingVarGetExtendedmasterconsdata(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef NDEBUG
#define GCGinferredPricingVarIsCoefVar(var)            (SCIPvarGetData(var)->data.inferredpricingvardata.iscoefvar)
#else
/* returns whether the inferred pricing variable is a coefficient variable */
GCG_EXPORT
SCIP_Bool GCGinferredPricingVarIsCoefVar(
   SCIP_VAR*             var                 /**< SCIP variable structure */
   );
#endif

#ifdef __cplusplus
}


#endif
/** @} */
#endif
