/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*                  This file is part of the program and library             */
/*          GCG --- Generic Column Generation                                */
/*                  a Dantzig-Wolfe decomposition based extension            */
/*                  of the branch-cut-and-price framework                    */
/*         SCIP --- Solving Constraint Integer Programs                      */
/*                                                                           */
/* Copyright (C) 2010-2026 Operations Research, RWTH Aachen University       */
/*                         Zuse Institute Berlin (ZIB)                       */
/*                                                                           */
/*  Licensed under the Apache License, Version 2.0 (the "License");          */
/*  you may not use this file except in compliance with the License.         */
/*  You may obtain a copy of the License at                                  */
/*                                                                           */
/*      http://www.apache.org/licenses/LICENSE-2.0                           */
/*                                                                           */
/*  Unless required by applicable law or agreed to in writing, software      */
/*  distributed under the License is distributed on an "AS IS" BASIS,        */
/*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. */
/*  See the License for the specific language governing permissions and      */
/*  limitations under the License.                                           */
/*                                                                           */
/*  You should have received a copy of the Apache-2.0 license                */
/*  along with GCG; see the file LICENSE. If not visit gcg.or.rwth-aachen.de.*/
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "symmetry/pub_automorphism.h"
#include "nauty/nausparse.h"
#include "scip/scip.h"
#include <vector>

struct NautyData
{
   void* staticuserdata;
   void (*staticfhook)(void*, unsigned int, const unsigned int*);
   statsblk* stats;
   unsigned int searchnodelimit;
   unsigned int generatorlimit;
   bool terminate;
};

static TLS_ATTR NautyData nautydata;
static TLS_ATTR DEFAULTOPTIONS_SPARSEGRAPH(options);

struct struct_graph_data
{
   SCIP* scip_;
   statsblk stats;
   sparsegraph graph;
   int* ptn;
   int* orbits;
   std::vector<std::vector<int>> edges;
   size_t nedges;
};

static
void nautyhook(
   int                   count,              /**< ordinal of this generator (starts with 1) */
   int*                  perm,               /**< the generator */
   int*                  orbits,             /**< orbits generated by the group found so far */
   int                   numorbits,          /**< number of orbits */
   int                   stabvertex,         /**< stabilizing vertex */
   int                   n                   /**< number of vertices */
   )
{
   if( nautydata.terminate
      || (nautydata.generatorlimit > 0 && (unsigned int)count >= nautydata.generatorlimit)
      || (nautydata.searchnodelimit > 0 && nautydata.stats->numnodes >= nautydata.searchnodelimit))
      return;
   else
      (*nautydata.staticfhook)(nautydata.staticuserdata, n, (unsigned int*)perm);
}

SCIP_RETCODE struct_graph::init(
   SCIP* scip,
   int nvertices
   )
{
   graphdata = new struct_graph_data();
   graphdata->scip_ = scip;
   options.userautomproc = &nautyhook;
   options.defaultptn = FALSE;
   SG_INIT(graphdata->graph);
   DYNALLOC1(size_t, graphdata->graph.v, graphdata->graph.vlen, nvertices, "malloc");
   DYNALLOC1(int, graphdata->graph.d, graphdata->graph.dlen, nvertices, "malloc");
   graphdata->graph.e = NULL;
   graphdata->graph.w = NULL;
   graphdata->graph.nv = nvertices;
   SCIP_CALL( SCIPallocBlockMemoryArray(scip, &graphdata->ptn, nvertices) );
   graphdata->edges.resize(nvertices);
   graphdata->nedges = 0;
   return SCIP_OKAY;
}

SCIP_RETCODE struct_graph::destroy(
   )
{
   SCIPfreeBlockMemoryArray(graphdata->scip_, &graphdata->ptn, graphdata->graph.nv);
   SG_FREE(graphdata->graph);
   delete graphdata;
   return SCIP_OKAY;
}

void struct_graph::setColor(
   int vertex,
   int color
   )
{
   assert(vertex < graphdata->graph.nv);
   graphdata->ptn[vertex] = color;
}

void struct_graph::addEdge(
   int v1,
   int v2
   )
{
   assert(v1 < (int) graphdata->edges.size() && v2 < (int) graphdata->edges.size());
   graphdata->edges[v1].push_back(v2);
   graphdata->edges[v2].push_back(v1);
   graphdata->nedges+=2;
}

unsigned int struct_graph::getNVertices(
   )
{
   return graphdata->graph.nv;
}

SCIP_RETCODE struct_graph::findAutomorphisms(
   void* userdata,
   void (*fhook)(void*, unsigned int, const unsigned int*),
   unsigned int searchnodelimit,
   unsigned int generatorlimit
   )
{
   int i;
   int* lab = NULL;
   int* orbits = NULL;
   size_t edgeindex = 0;

   // edges
   DYNALLOC1(int, graphdata->graph.e, graphdata->graph.elen, graphdata->nedges, "malloc");
   graphdata->graph.nde = graphdata->nedges;
   assert((int) graphdata->edges.size() == graphdata->graph.nv);
   for( i = 0; i < graphdata->graph.nv; ++i)
   {
      graphdata->graph.d[i] = (int)graphdata->edges[i].size();
      graphdata->graph.v[i] = edgeindex;
      for( auto v : graphdata->edges[i] )
      {
         assert(edgeindex < graphdata->graph.nde);
         graphdata->graph.e[edgeindex++] = v;
      }
   }

   // set coloring
   SCIP_CALL( SCIPallocBufferArray(graphdata->scip_, &lab, graphdata->graph.nv) );
   for( i = 0; i < graphdata->graph.nv; ++i )
      lab[i] = i;
   SCIPsortIntInt(graphdata->ptn, lab, graphdata->graph.nv);
   for( i = 0; i < graphdata->graph.nv; ++i )
   {
      if( i + 1 == graphdata->graph.nv ||  graphdata->ptn[i] != graphdata->ptn[i+1] )
         graphdata->ptn[i] = 0;
      else
         graphdata->ptn[i] = 1;
   }

   SCIP_CALL( SCIPallocBufferArray(graphdata->scip_, &orbits, graphdata->graph.nv) );

   nautydata.staticfhook = fhook;
   nautydata.staticuserdata = userdata;
   nautydata.stats = &graphdata->stats;
   nautydata.terminate = false;
   sparsenauty(&graphdata->graph, lab, graphdata->ptn, orbits, &options, &graphdata->stats, NULL);

   SCIPfreeBufferArray(graphdata->scip_, &orbits);
   SCIPfreeBufferArray(graphdata->scip_, &lab);

   return SCIP_OKAY;
}

void struct_graph::terminateSearch()
{
   nautydata.terminate = true;
}

extern "C"
void GCGgetNautyName(char* buffer, int len)
{
   SCIPsnprintf(buffer, len, "Nauty %d.%d.%d", NAUTYVERSIONID/10000, (NAUTYVERSIONID%10000)/1000, (NAUTYVERSIONID%1000)/10);
}
