#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Feb 28 12:25:35 2018

@author: witt
"""
#!/usr/bin/python
import matplotlib  as mpl
mpl.use('Agg')

import os
import sys
import glob
import re
import argparse
import subprocess
from subprocess import call, DEVNULL, STDOUT, check_call


import pandas as pd
import matplotlib.pyplot as plt
import numpy as np

from matplotlib.backends.backend_pdf import PdfPages

DIR, FILENAME = os.path.split(__file__)
params = {}

#os.chdir(os.path.dirname(os.path.realpath(__file__)))

def parse_arguments(args):
    """
    Parse the command-line arguments
    :param args: Command-line arguments provided during execution
    :return: Parsed arguments
    """
    parser = argparse.ArgumentParser()

    parser.add_argument('-tm', '--time', type=int, default=22,
                        help='Column for solution time res-files')
    parser.add_argument('-st', '--status', type=int, default=23,
                        help='Column for status in res-file')
    parser.add_argument('-m', '--min', type=float, default=1.0,
                        help='Minimum solution times (solution times are set to maximum of solution times and this value)')
    parser.add_argument('--max', type=float, default=None,
                        help='Maximum solution times (should not be used, but can be to remove very long runners that make the plot hard to read)')
    parser.add_argument('-ss', '--stepsize', type=float, default=0.01,
                        help='Step size used in plot (note that creating the plot can take long when stepsize is too small)')
    parser.add_argument('-set', '--settingloc', type=int, default=-3,
                        help='give the position (counting from 0) of your settings name in the file name after it has been split by dots.')
    parser.add_argument('-l', '--log', type=bool, default=False,
                        help='Should a logarithmic scale be used?')

    parser.add_argument('-to', '--timeouts', type=bool, default=False,
                        help='Should time outs be counted?')

    parser.add_argument('-nl', '--nodelimits', type=bool, default=False,
                        help='Should node limits be counted?')

    parser.add_argument('-ab', '--aborts', type=bool, default=False,
                        help='Should aborts be counted?')

    parser.add_argument('-f', '--filename', type=str, default='performance_profile.pdf',
                        help='Name of out file')

    parser.add_argument('--outdir', type=str,
                        default="plots/",
                        help="output directory (default: 'plots/') - must end with '/'")

    parser.add_argument('--cluster',
                        action="store_true",
                        help='Was your out file generated by a cluster testrun?')

    parser.add_argument('resfiles', nargs='+',
                        help='Name of .res files to be used for the plot (only instances solved in all .res files are used)')

    parsed_args = parser.parse_args(args)
    return parsed_args


def set_params(args):
    """
    Set the global parame<rs from the parsed command-line arguments
    :param args: parsed command-line arguments
    :return:
    """
    params['time'] = args.time
    params['status'] = args.status
    params['min'] = args.min
    params['max'] = args.max
    params['stepsize'] = args.stepsize
    params['log'] = args.log
    params['timeouts'] = args.timeouts
    params['nodelimits'] = args.nodelimits
    params['aborts'] = args.aborts
    params['out'] = args.filename
    params['cluster'] = args.cluster
    params['outdir'] = args.outdir
    params['settingloc'] = args.settingloc

def main():
    """Entry point when calling this script"""
    args = sys.argv[1:]
    parsed_args = parse_arguments(args)
    set_params(parsed_args)

    if not os.path.exists(params['outdir']):
        os.makedirs(params['outdir'])

    df=pd.DataFrame()
    resfiles = parsed_args.resfiles
    # Check ending of file, if it is an out-file, first convert it, else just use it
    resorout = -1
    i = 0
    while i < len(resfiles):
        if not os.path.isfile(resfiles[i]):
            print("File number {} could not be found. Terminating.".format(i+1))
            exit()
        if resfiles[i].endswith(".res") and not resorout == 1:
            print("Interpreting your input as .res file")
            resorout = 0
        elif resfiles[i].endswith(".out") and not resorout == 0:
            resorout = 1
            if params['cluster'] or ("clustor" in resfiles[i].split('.')):
                print("Converting '{}' to a resfile (cluster).".format(resfiles[i]))
                subprocess.check_call(['./misc/out2res_cluster.sh',resfiles[i]], stdout=DEVNULL, stderr=STDOUT)
            else:
                print("Converting '{}' to a resfile (client).".format(resfiles[i]))
                subprocess.check_call(['./misc/out2res.sh',resfiles[i]], stdout=DEVNULL, stderr=STDOUT)
            resfiles[i] = re.sub('.out$', '.res',resfiles[i])
        else:
            print("Unsupported file or mixed file formats. Terminating.")
            exit()
        i += 1
    for resfile in resfiles:
        files = list(filter(os.path.isfile, glob.glob(resfile)))
        files.sort(key=lambda x: os.path.getmtime(x))
        dftmp = pd.DataFrame()
        if len(files) > 0:
            # for this case, most probably, the position of params[time] and params[status] was incorrect
            #if len(dftmp.dropna(axis=0, how="any").values) == 0:
            with open(resfile) as f:
                f.readline()
                header = f.readline().replace(" ", "").strip().split('|')
            params['time'] = len(header) - 1 - header[::-1].index('Time')
            params['status'] = len(header) - 1 - header[::-1].index('')
            dftmp = pd.read_csv(
                    files[0],
                    skiprows=3, index_col = 0, delim_whitespace=True, skipfooter=10, usecols=[0,params['time'],params['status']], names=['instance',(resfile,'time'),(resfile,'status')], engine='python')
            #print("res file    :", files[0])
            #dftmp = pd.read_csv(
            #        files[0],
            #        skiprows=3, index_col = 0, delim_whitespace=True, skipfooter=10, usecols=[0,params['time'],params['status']], names=['instance',(resfile,'time'),(resfile,'status')], engine='python')
            # this is a workaround for a bug
            if len(dftmp.columns) > 2:
                dftmp = pd.read_csv(
                    files[0],
                    skiprows=3, index_col = 0, delim_whitespace=True, skipfooter=10, usecols=[params['time']-1,params['status']-1], names=[(resfile,'time'),(resfile,'status')], engine='python')


        else:
            failed = True
            print("WARNING      : No res-file", resfile)
        df = pd.merge(left=dftmp,right=df,how='left',left_index=True, right_index=True)
        df = pd.concat([dftmp,df], axis=0, join='outer',sort=False)

    if df.empty:
        print("Fatal: Could not read files or files empty.\nTerminating.")
        exit()
    for resfile in resfiles:
        statusfilter = ["ok","solved"]
        if params["timeouts"]:
            statusfilter.append("timeout")
        if params["nodelimits"]:
            statusfilter.append("nodelimit")
        if params["aborts"]:
            statusfilter.append("abort")

        df = df[df[resfile,"status"].isin(statusfilter)]

        df[resfile,'time'] = df[resfile,'time'].apply(lambda x: max(x, params['min']))
        if params['max'] is not None:
            df[resfile,'time'] = df[resfile,'time'].apply(lambda x: min(x, params['max']))

    df["best"] = df[[(resfile, 'time') for resfile in resfiles]].min(axis=1)

    for resfile in resfiles:
        df[resfile,"performance"] = df[resfile,"time"]/df["best"]

    df["rmax"] = df[[(resfile, 'performance') for resfile in resfiles]].max(axis=1)
    rmax = df["rmax"].max()

    def perplot(resfile, t):
        return 1.0*len(df[df[resfile,"performance"] <= t]) / len(df)
    try: xx = np.arange(1.0,rmax+params['stepsize'],params['stepsize'])
    except ValueError:
        print("Fatal: No data to plot. Either your two runs have no optimally solved instances in common, \n" +
              "       or your columns (time and status) were incorrect. See --help for more information.")
        print("Terminating.")
        exit()
    f, ax = plt.subplots()

    for resfile in resfiles:
        try:
            setting = resfile.split(".")[params['settingloc']]#.split('-')[0]
        except:
            print("Warning: Your file naming scheme does not match the default GCG scheme.\n         The names of your settings could not be determined.")
            setting = "default"

        yy = np.array([perplot(resfile,x) for x in xx])
        if params['log']:
            ax.semilogx(xx,yy,label=setting)
        else:
            ax.plot(xx,yy,label=setting, linewidth="1.1")

#        ax = df.plot(kind='line',y=(resfile,"per"), ax=ax);
    plt.ylim([0.0, 1.01])
    plt.legend(loc='lower right')

    pp = PdfPages(params['outdir'] + params['out'])
    plt.title("Performance Profile")
    plt.xlabel("Relative slowdown compared with best performing run", size='small')
    plt.ylabel("Relative number of instances", size='small')
    plt.savefig(pp, format='pdf')
    pp.close()


# Calling main script
if __name__ == '__main__':
    main()
