/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*                                                                           */
/*          This file is part of the program and software framework          */
/*                    UG --- Ubquity Generator Framework                     */
/*                                                                           */
/*  Copyright Written by Yuji Shinano <shinano@zib.de>,                      */
/*            Copyright (C) 2021-2024 by Zuse Institute Berlin,              */
/*            licensed under LGPL version 3 or later.                        */
/*            Commercial licenses are available through <licenses@zib.de>    */
/*                                                                           */
/* This code is free software; you can redistribute it and/or                */
/* modify it under the terms of the GNU Lesser General Public License        */
/* as published by the Free Software Foundation; either version 3            */
/* of the License, or (at your option) any later version.                    */
/*                                                                           */
/* This program is distributed in the hope that it will be useful,           */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             */
/* GNU Lesser General Public License for more details.                       */
/*                                                                           */
/* You should have received a copy of the GNU Lesser General Public License  */
/* along with this program.  If not, see <http://www.gnu.org/licenses/>.     */
/*                                                                           */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**@file    bbParaParamSet.cpp
 * @brief   Parameter set for UG framework.
 * @author  Yuji Shinano
 *
 *
 *
 */

/*---+----1----+----2----+----3----+----4----+----5----+----6----+----7----+----8----+----9----+----0----+----1----+----2*/


#include <string>
#include <map>
#include <cstdio>
#include <cstring>
#include <fstream>
#include <cfloat>
#include <climits>
#include <cassert>
#include "ug/paraComm.h"
#include "bbParaParamSet.h"

using namespace UG;

// ParaParam *BbParaParamSet::paraParams[ParaParamsSize];

BbParaParamSet::BbParaParamSet(
      size_t inNParaParams
      )
#ifdef _COMM_MPI_WORLD
      : ParaParamSetMpi(inNParaParams)
#else
      : ParaParamSetTh(inNParaParams)
#endif
{

  /** bool params */
   paraParams[LogSubtreeInfo] = new ParaParamBool(
         "LogSubtreeInfo",
         "# Control output of subtree log [Default value: FALSE]",
         false);
   paraParams[OutputTabularSolvingStatus] = new ParaParamBool(
         "OutputTabularSolvingStatus",
         "# Control output of tabular solving statues [Default value: TRUE]",
         true);
   paraParams[DeterministicTabularSolvingStatus] = new ParaParamBool(
         "DeterministicTabularSolvingStatus",
         "# Control output of tabular solving status using deterministic time (only valid when Deterministic = TRUE)  [Default value: FALSE]",
         false);
   paraParams[UseRootNodeCuts] = new ParaParamBool(
         "UseRootNodeCuts",
         "# Use root node cuts in ParaInstance, that is, the instance data for solvers has the root node cuts of the original problem. [Default value: FALSE]",
         false);
   paraParams[TransferLocalCuts] = new ParaParamBool(
         "TransferLocalCuts",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes local cuts generated by solvers or not. TRUE means it is included. [Default value: TRUE]",
         true);
   paraParams[TransferConflictCuts] = new ParaParamBool(
         "TransferConflictCuts",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes conflict cuts generated by a solver s or not. TRUE means it is included. [Default value: FALSE]",
         false);
   paraParams[TransferConflicts] = new ParaParamBool(
         "TransferConflicts",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes conflicts generated by a solver s or not. TRUE means it is included. [Default value: TRUE]",
         true);
   paraParams[TransferBranchStats] = new ParaParamBool(
         "TransferBranchStats",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes branching status generated by a solver s or not. TRUE means it is included. [Default value: TRUE]",
         true);
   paraParams[TransferVarValueStats] = new ParaParamBool(
         "TransferVarValueStats",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes var-value status generated by a solver s or not. TRUE means it is included. [Default value: FALSE]",
         false);
   paraParams[TransferBendersCuts] = new ParaParamBool(
         "TransferBendersCuts",
         "# Indicate if subproblem information (in ParaDiffSubproblem) includes benders cuts generated by a solver s or not. TRUE means it is included. [Default value: FALSE]",
         false);
   paraParams[CheckEffectOfRootNodePreprocesses] = new ParaParamBool(
         "CheckEffectOfRootNodePreprocesses",
         "# Check effect of root node preprocesses. [Default value: FALSE]",
         false);
   paraParams[CollectOnce] = new ParaParamBool(
         "CollectOnce",
         "# Control whether or not all open nodes are collected to load coordinator after ramp-up. [Default value: TRUE]",
         true);
   paraParams[ProvingRun] = new ParaParamBool(
         "ProvingRun",
         "# Control to run for proving or not. [Default value: FALSE]",
         false);
   paraParams[SetAllDefaultsAfterRacing] = new ParaParamBool(
         "SetAllDefaultsAfterRacing",
         "# Set all parameter values to default after racing. [Default value: FALSE]",
         false);
   paraParams[DistributeBestPrimalSolution] = new ParaParamBool(
         "DistributeBestPrimalSolution",
         "# Control if the best primal solution is distributed or not. [Default value: TRUE]",
         true);
   paraParams[LightWeightRootNodeProcess] = new ParaParamBool(
         "LightWeightRootNodeProcess",
         "# Control if light weight root node processing is applied. [Default value: FALSE]",
         false);
   paraParams[RacingStatBranching] = new ParaParamBool(
         "RacingStatBranching",
         "# Control if racing status branching is applied. [Default value: TRUE]",
         true);
   paraParams[IterativeBreakDown] = new ParaParamBool(
         "IterativeBreakDown",
         "# Control if iterative break down is applied. [Default value: FALSE]",
         false);
   paraParams[NoPreprocessingInLC] = new ParaParamBool(
         "NoPreprocessingInLC",
         "# Control if preprocessing is disabled in LoadCoordinator. [Default value: FALSE (enabled)]",
         false);
   paraParams[NoUpperBoundTransferInRacing] = new ParaParamBool(
         "NoUpperBoundTransferInRacing",
         "# Disable transfer of upper bound to the other solvers in racing. [Default value: FALSE (transfer)]",
         false);
   paraParams[MergeNodesAtRestart] = new ParaParamBool(
         "MergeNodesAtRestart",
         "# Merge nodes at restart or not. [Default value: FALSE]",
         false);
   paraParams[NChangeIntoCollectingModeNSolvers] = new ParaParamBool(
         "NChangeIntoCollectingModeNSolvers",
         "# Control whether or not to NChangeIntoCollectingMode to the number of solvers/2  [Default value: FALSE]",
         false);
   paraParams[EventWeightedDeterministic] = new ParaParamBool(
         "EventWeightedDeterministic",
         "# Enable event weighted deterministiv run (each event has its weight of deterministic time) [Default value: FALSE]",
         false);
   paraParams[NoSolverPresolvingAtRoot] = new ParaParamBool(
         "NoSolverPresolvingAtRoot",
         "# Disable presolving at root node, for distributed memory version this should be set to FALSE  [Default value: FALSE (enable presolving)]",
         false);
   paraParams[NoSolverPresolvingAtRootDefaultSet] = new ParaParamBool(
         "NoSolverPresolvingAtRootDefaultSet",
         "# Disable presolving at root and set default parameter settings: for distributed memory version this should be set to FALSE  [Default value: FALSE (enable presolving)]",
         false);
   paraParams[NoAggressiveSeparatorInRacing] = new ParaParamBool(
         "NoAggressiveSeparatorInRacing",
         "# Disable aggressive separator settings in racing  [Default value: TRUE]",
         true);
   paraParams[AllBoundChangesTransfer] = new ParaParamBool(
         "AllBoundChangesTransfer",
         "# Transfer bound changes not only for branching variables, but for all variables. [Default value: TRUE]",
         true);
   paraParams[NoAllBoundChangesTransferInRacing] = new ParaParamBool(
         "NoAllBoundChangesTransferInRacing",
         "# Disable transfer of all bound changes when the stage is in racing, even if AllBoundChangesTransfer=TRUE [Default value: FALSE]",
         false);
   paraParams[BreakFirstSubtree] = new ParaParamBool(
         "BreakFirstSubtree",
         "# Break the first subtree [Default value: FALSE]",
         false);
   paraParams[InitialNodesGeneration] = new ParaParamBool(
         "InitialNodesGeneration",
         "# Run to generate initial nodes [Default value: FALSE]",
         false);
   paraParams[RestartRacing] = new ParaParamBool(
         "RestartRacing",
         "# Restart racing ramp-up, when primal solution was updated in racing [Default value: FALSE]",
         false);
   paraParams[CheckFeasibilityInLC] = new ParaParamBool(
         "CheckFeasibilityInLC",
         "# Check solution feasiblity in LoadCoordinator [Default value: FALSE]",
         false);
   paraParams[ControlCollectingModeOnSolverSide] = new ParaParamBool(
         "ControlCollectingModeOnSolverSide",
         "# collection mode can be controlled on solver side [Default value: FALSE]",
         false);
   paraParams[CleanUp] = new ParaParamBool(
         "CleanUp",
         "# Clean up run for reducing check-pointing file [Default value: FALSE]",
         false);
   paraParams[DualBoundGainTest] = new ParaParamBool(
         "DualBoundGainTest",
         "# Test dual bound gain in Solver [Default value: FALSE]",
         false);
   paraParams[GenerateReducedCheckpointFiles] = new ParaParamBool(
          "GenerateReducedCheckpointFiles",
          "# The parallel solver is used only to generate reduced checkpoint files. [Default value: FALSE]",
          false);
   paraParams[OutputPresolvedInstance] = new ParaParamBool(
          "OutputPresolvedInstance",
          "# Output presolved instance. This is useful to verify the instance data at restart [Default value: FALSE]",
          false);
   paraParams[CommunicateTighterBoundsInRacing] = new ParaParamBool(
          "CommunicateTighterBoundsInRacing",
          "# Communicate tighter bounds of variables in racing. [Default value: TRUE]",
          true);
   paraParams[KeepRacingUntilToFindFirstSolution] = new ParaParamBool(
          "KeepRacingUntilToFindFirstSolution",
          "# Keep racing until first feasible solution was found. [Default value: FALSE]",
          false);
   paraParams[AllowTreeSearchRestart] = new ParaParamBool(
          "AllowTreeSearchRestart",
          "# Allow tree restart in solver [Default value: FALSE]",
          false);
   paraParams[OmitInfeasibleTerminationInRacing] = new ParaParamBool(
          "OmitInfeasibleTerminationInRacing",
          "# Omit infeasible terminations during racing [Default value: FALSE]",
          false);
   paraParams[WaitTerminationOfThreads] = new ParaParamBool(
          "WaitTerminationOfThreads",
          "# In case of racing termination wait for all threads to terminate [Default value: TRUE]",
          true);
   paraParams[EnhancedFinalCheckpoint] = new ParaParamBool(
          "EnhancedFinalCheckpoint",
          "# Enhanced checkpoint files are created, in which more nodes than normal checkpoint are included [Default value: FALSE]",
          false);

   /** int params */
   paraParams[RampUpPhaseProcess] = new ParaParamInt(
         "RampUpPhaseProcess",
         "# Ramp-up phase process: 0 - normal process, 1 - racing process, 2 - rebuild tree after racing, 3 - self-split process [Default value: 2]",
         2,
         0,
         3);
   paraParams[NChangeIntoCollectingMode] = new ParaParamInt(
         "NChangeIntoCollectingMode",
         "# Number of nodes in LoadCoordinator to decide to change into collection mode [Default value: 1][0, INT_MAX]",
         1,
         0,
         INT_MAX);
   paraParams[NodeTransferMode] = new ParaParamInt(
         "NodeTransferMode",
         "# Node transfer mode: 0 : best estimate node transfer, 1 : best bound node transfer [Default value: 1]",
         1,
         0,
         1);
   paraParams[MinNumberOfCollectingModeSolvers] = new ParaParamInt(
         "MinNumberOfCollectingModeSolvers",
         "# Minimum number of Solvers that can be in collection mode. [Default value: 1]",
         1,
         1,
         INT_MAX);
   paraParams[MaxNumberOfCollectingModeSolvers] = new ParaParamInt(
         "MaxNumberOfCollectingModeSolvers",
         "# Maximum number of Solvers that can be in collection mode : -1 : no limit, 0 : half of the number of Solvers, n > 0 : n [Default value: 0]",
         0,
         -1,
         INT_MAX);
   paraParams[SolverOrderInCollectingMode] = new ParaParamInt(
         "SolverOrderInCollectingMode",
         "# The order to send request messages in collection mode: -1 : no ordering, 0 : ordered by best dual bound value, 1 : ordered by number of nodes left, 2 : choose alternatively the best bound and the number of nodes orders  [Default value: 0]",
         0,
         -1,
         2);
   paraParams[RacingRampUpTerminationCriteria] = new ParaParamInt(
         "RacingRampUpTerminationCriteria",
         "# Racing ramp-up termination criteria : 0 : stop at the number of nodes left reached, 1 : stop at time limit, 2: : stop at the Solver with the best dual bound value has a certain number of nodes, 3 : adaptive(node first), 4 : adaptive (time first):  [Default value: 4]",
         5,
         0,
         7);
   paraParams[StopRacingNumberOfNodesLeft] = new ParaParamInt(
         "StopRacingNumberOfNodesLeft",
         "# The number of nodes left in a solver to stop racing : [1, INT_MAX]:  [Default value: 300]",
         300,
         1,
         INT_MAX);
   paraParams[NumberOfNodesKeepingInRootSolver] = new ParaParamInt(
         "NumberOfNodesKeepingInRootSolver",
         "# The number of nodes left in the root solver to stop keeping nodes. 0: no keeping nodes : [0, INT_MAX]:  [Default value: 0]",
         0,
         0,
         INT_MAX);
   paraParams[NumberOfInitialNodes] = new ParaParamInt(
         "NumberOfInitialNodes",
         "# The number of nodes left in all solvers to stop generating initial nodes : [1, INT_MAX]:  [Default value: 300]",
         300,
         1,
         INT_MAX);
   paraParams[MaxNRacingParamSetSeed] = new ParaParamInt(
         "MaxNRacingParamSetSeed",
         "# The maximum number of seeds for racing parameter set : [1, INT_MAX]:  [Default value: 64]",
         64,
         1,
         INT_MAX);
   paraParams[TryNVariablegOrderInRacing] = new ParaParamInt(
         "TryNVariablegOrderInRacing",
         "# The number of variable order tries in racing ramp-up : [1, INT_MAX]:  [Default value: 100]",
         100,
         1,
         INT_MAX);
   paraParams[TryNBranchingOrderInRacing] = new ParaParamInt(
         "TryNBranchingOrderInRacing",
         "# The number of branching order tries in racing ramp-up : [1, INT_MAX]:  [Default value: 100]",
         100,
         1,
         INT_MAX);
   paraParams[NEvaluationSolversToStopRacing] = new ParaParamInt(
          "NEvaluationSolversToStopRacing",
          "# The number of evaluation solvers to stop racing. ( -1 stops at all of the solvers, 0 stop at half of the solvers ) : [-1, INT_MAX]:  [Default value: -1]",
          -1,
          -1,
          INT_MAX);
   paraParams[NMaxCanditatesForCollecting] = new ParaParamInt(
          "NMaxCanditatesForCollecting",
          "# The maximum number of candidates to be in collection mode solvers, when system goes in to collecting mode : [1, INT_MAX]:  [Default value: 10]",
          10,
          1,
          INT_MAX);
   paraParams[NSolverNodesStartBreaking] = new ParaParamInt(
          "NSolverNodesStartBreaking",
          "# The number of nodes left in a solver to start breaking. : [0, INT_MAX]:  [Default value: 0: no breaking]",
          0,
          0,
          INT_MAX);
   paraParams[NStopBreaking] = new ParaParamInt(
          "NStopBreaking",
          "# The number of nodes left in the system to stop breaking. : [0, INT_MAX]:  [Default value: 0: no breaking]",
          0,
          0,
          INT_MAX);
   paraParams[NTransferLimitForBreaking] = new ParaParamInt(
          "NTransferLimitForBreaking",
          "# The maximum number of nodes transferred from a solver, when system is in breaking mode. : [1, INT_MAX]:  [Default value: 100]",
          100,
          1,
          INT_MAX);
   paraParams[NStopSolvingMode] = new ParaParamInt(
         "NStopSolvingMode",
         "# Stop solving mode if number of solved nodes is less than NStopSolvingMode. 0 means to no stop solving. [Default: 3][-1, INT_MAX]",
         3,
         -1,
         INT_MAX);
   paraParams[NCollectOnce] = new ParaParamInt(
         "NCollectOnce",
         "# Number of nodes collected once to load coordinator. The value -1 means all. 0 means # solvers * 5. [Default: 0][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[AggressivePresolveDepth] = new ParaParamInt(
         "AggressivePresolveDepth",
         "# Depth to try presolving in the other solvers. Offset value will be added. -1: no aggressive presolving. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[AggressivePresolveStopDepth] = new ParaParamInt(
         "AggressivePresolveStopDepth",
         "# Depth to stop aggressive presolving. This value is only valid for AggressivePresolveDepth > 0. [Default: 8][1, INT_MAX]",
         8,
         1,
         INT_MAX);
   paraParams[BigDualGapSubtreeHandling] = new ParaParamInt(
         "BigDualGapSubtreeHandling",
         "# Big dual gap subtree handling. 0 : throw away, 1 : send back to load coordinator [Default: 0][0, 1]",
         0,
         0,
         1);
   paraParams[InstanceTransferMethod] = new ParaParamInt(
         "InstanceTransferMethod",
         "# Instance data transfer method. 0 : on memory, 1 : by file, 2 : by original data [Default: 0][0, 2]",
         0,
         0,
         2);
   paraParams[KeepNodesDepth] = new ParaParamInt(
         "KeepNodesDepth",
         "# Disable transfer and keep nodes depth from sub-MIP root less than this value. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[NoAlternateSolving] = new ParaParamInt(
         "NoAlternateSolving",
         "# Alternate solving is prohibited when the specified value is greater than zero. If the number of nodes left is less than the value, alternate solving is performed. [Default: 100][0, INT_MAX]",
         100,
         0,
         INT_MAX);
   paraParams[NNodesTransferLogging] = new ParaParamInt(
         "NNodesTransferLogging",
         "# The number of nodes transferred is logged when the specified value greater than zero. The specified value indicates the frequency for the logging. [Default: 0][0, INT_MAX]",
         0,
         0,
         INT_MAX);
   paraParams[NIdleSolversToTerminate] = new ParaParamInt(
         "NIdleSolversToTerminate",
         "# The number of idle solvers to terminate forcibly. Value < 0 means no termination depending of the number of idle solvers.  [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[FinalCheckpointNSolvers] = new ParaParamInt(
         "FinalCheckpointNSolvers",
         "# The maximum number of solvers whose nodes are collected at the final checkpoint. The value = -1 means all solvers nodes are collected. [Default: 10][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[NMergingNodesAtRestart] = new ParaParamInt(
         "NMergingNodesAtRestart",
         "# The number of nodes trying to merge at restart. -1 : that to try to merge all nodes. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[NBoundChangesOfMergeNode] = new ParaParamInt(
         "NBoundChangesOfMergeNode",
         "# How many number of bound changes in a subproblem is needed to be merged.  -1 : set automatically. 0 : everything. [Default: -1][-1, INT_MAX]",
         -1,
         -1,
         INT_MAX);
   paraParams[NNodesToKeepInCheckpointFile] = new ParaParamInt(
         "NNodesToKeepInCheckpointFile",
         "# Number of the best nodes to keep in checkpoint file, that is not be processed. This parameter is only valid in restarting with checkpoint file. [Default: 0][0, INT_MAX]",
         0,
         0,
         INT_MAX);
   paraParams[NMaxRacingBaseParameters] = new ParaParamInt(
         "NMaxRacingBaseParameters",
         "# Maximum number of racing base parameter configurations. File started from 1 to NMaxRacingBaseParameters. [Default: 0][0, 99999]",
         0,
         0,
         99999);
   paraParams[NBoundChangesForTransferNode] = new ParaParamInt(
         "NBoundChangesForTransferNode",
         "# Number of bound changes needed to transfer a branch node. -1 : no restrictions. [Default: -1][-1, 99999]",
         -1,
         -1,
         99999);
   paraParams[OmitTerminationNSolutionsInRacing] = new ParaParamInt(
         "OmitTerminationNSolutionsInRacing",
         "# Omit racing termination until the number of incumbent solutions is obtained. 1 : at least one feasible solution is needed. [Default: 0][0, 99999]",
         0,
         0,
         99999);
   paraParams[NEagerToSolveAtRestart] = new ParaParamInt(
         "NEagerToSolveAtRestart",
         "# Number of best nodes, which are eager to be solved at restart. 0 : all nodes. [Default: 0][0, 99999]",
         0,
         0,
         999999);
   paraParams[SelfSplitTreeDepth] = new ParaParamInt(
         "SelfSplitTreeDepth",
         "# depth of branch-and-bound to open nodes for self-split [Default value: 0]",
         0,
         0,
         INT_MAX);
   paraParams[LightWeightNodePenartyInCollecting] = new ParaParamInt(
         "LightWeightNodePenartyInCollecting",
         "# The number of light weight nodes in collecting mode, which can be considered as one heavy weight node [Default value: 10]",
         10,
         1,
         INT_MAX);
   paraParams[EnhancedCheckpointInterval] = new ParaParamInt(
         "EnhancedCheckpointInterval",
         "# Specify an interval to create checkpoint files. It is specified by the number of solvers, in which open nodes are collected. [Default value: 10]",
         10,
         1,
         INT_MAX);

   /** longint params */

   /** real params */
   paraParams[MultiplierForCollectingMode] = new ParaParamReal(
         "MultiplierForCollectingMode",
         "# Multiplier for mp in collection mode. When the number of good nodes becomes greater than mp * NChangeIntoCollectingMode, stop collecting [Default value: 2.0][1.0, DBL_MAX]",
         2.0,
         1.0,
         DBL_MAX);
   paraParams[MultiplierToDetermineThresholdValue] = new ParaParamReal(
         "MultiplierToDetermineThresholdValue",
         "# Multiplier for mth to determine threshold value. When the number of nodes left greater than mth * (meanRootNodeTime/ (meanNodeTime except root)), a Solver starts sending nodes in collection mode [Default: 1.5][1.0, DBL_MAX]",
         1.5,
         1.0,
         DBL_MAX);
   paraParams[BgapCollectingMode] = new ParaParamReal(
         "BgapCollectingMode",
         "# Value boundgap_p to identify good nodes, (bound value of a node - best bound value )/ (best bound value ) < gap_p is treated as good node. [Default: 0.4][0.0,1.0]",
         0.4,
         0.0,
         1.0);
   paraParams[MultiplierForBgapCollectingMode] = new ParaParamReal(
         "MultiplierForBgapCollectingMode",
         "# Define when a Solver in collection mode leaves it: (the solver's best dual bound value - g.b.d.b.v )/g.b.d.b.v > gap_p* (MultiplierForBgapCollectingMode) [Default:10.0][1.0, 10.0]",
         10.0,
         1.0,
         10.0);
   paraParams[ABgapForSwitchingToBestSolver] = new ParaParamReal(
         "ABgapForSwitchingToBestSolver",
         "# Define when a Solver in collection mode leaves it: when the absolute dual gap to the best solver is greater than this value and the best solver is not in collection mode (Don't set 0.0)  [Default:1.0][0.0, DBL_MAX]",
         1.0,
         0.0,
         DBL_MAX);
   paraParams[BgapStopSolvingMode] = new ParaParamReal(
         "BgapStopSolvingMode",
         "# Define when a solver is stopped and all generated nodes except the root nodes are discarded, when ( its local best dual bound value - LC.b.d.b.v ) / LC.b.d.b.v > BgapStopSolvingMode. [Default: 0.33][0.0, 10.0]",
         0.33,
         0.0,
         10.0);
   paraParams[StopRacingTimeLimit] = new ParaParamReal(
         "StopRacingTimeLimit",
         "# The time limit until racing is stopped. [Default: 720.0] [1.0, DBL_MAX]",
         720.0,
         1.0,
         DBL_MAX);
   paraParams[StopRacingTimeLimitMultiplier] = new ParaParamReal(
         "StopRacingTimeLimitMultiplier",
         "# An extending time limit multiplier in case nodes left is less than stop-racing number of nodes left. [Default: 50.0] [1.0, DBL_MAX]",
         50.0,
         1.0,
         DBL_MAX);
   paraParams[StopRacingNumberOfNodesLeftMultiplier] = new ParaParamReal(
         "StopRacingNumberOfNodesLeftMultiplier",
         "# The multiplier for the limit of the number of nodes left to stop racing. [Default: 20.0] [1.0, DBL_MAX]",
         3.0,
         1.0,
         DBL_MAX);
   paraParams[TimeToIncreaseCMS] = new ParaParamReal(
         "TimeToIncreaseCMS",
         "# Timelimit for empty node pool. When node pool stays empty longer than this time, then the number of collection mode solvers will be increased. [Default: 10.0] [1.0, DBL_MAX]",
         10.0,
         1.0,
         DBL_MAX);
   paraParams[TabularSolvingStatusInterval] = new ParaParamReal(
         "TabularSolvingStatusInterval",
         "# Time interval inbetween output of tabular solving status log [Default: 5.0] [0.0, DBL_MAX]",
         5.0,
         0.0,      // for debug, this is useful
         DBL_MAX);
   paraParams[RatioToApplyLightWeightRootProcess] = new ParaParamReal(
         "RatioToApplyLightWeightRootProcess",
         "# If the ratio between idle solvers divided by the total number of solvers is smaller than this parameter, then the light weight root node process is applied.  [Default: 0.5] [0.0, 1.0]",
         0.5,
         0.0,
         1.0);
   paraParams[MultiplierForBreakingTargetBound] = new ParaParamReal(
         "MultiplierForBreakingTargetBound",
         "# Solvers, which have the best dual bound nodes less than (the global best dual bound * this multiplier), become target of breaking solvers.  [Default: 1.03] [1.0, DBL_MAX]",
         1.03,
         1.0,
         DBL_MAX);
   paraParams[FixedVariablesRatioInMerging] = new ParaParamReal(
         "FixedVariablesRatioInMerging",
         "# Ratio of fixed variables when merging open nodes.  [Default: 0.9] [0.0, 1.0]",
         0.9,
         0.0,
         1.0);
   paraParams[AllowableRegressionRatioInMerging] = new ParaParamReal(
         "AllowableRegressionRatioInMerging",
         "# Permissible regression ratio of dual bound value in merging.  [Default: 0.0] [0.0, 1.0]",
         0.0,
         0.0,
         1.0);
   paraParams[CountingSolverRatioInRacing] = new ParaParamReal(
         "CountingSolverRatioInRacing",
         "# Ratio of status notified solvers required to determine the winner in racing. Note: a solver with a specific setting sometimes has a hard time to solve a root node and cannot send status message to the load coordinator.  [Default: 0.5] [0.0, 1.0]",
         0.5,
         0.0,
         1.0);
   paraParams[ProhibitCollectOnceMultiplier] = new ParaParamReal(
         "ProhibitCollectOnceMultiplier",
         "# When the racing winner has nodes less than (the number of Solvers)*(this value), CollectOnce should be prohibited.  [Default: 0.0] [0.0, DBL_MAX]",
         0.0,
         0.0,
         DBL_MAX);
   paraParams[TNodesTransferLogging] = new ParaParamReal(
         "TNodesTransferLogging",
         "# Log the number of transferred nodes in the specified time interavl. Logging is turned off when the specified value is nonpositive. . [Default: 2.0][-1.0, DBL_MAX]",
         2.0,
         -1.0,
         DBL_MAX);
   paraParams[RandomNodeSelectionRatio] = new ParaParamReal(
         "RandomNodeSelectionRatio",
         "# Frequency to select a node randomly from node pool in load coordinator.  [Default: 0.2][0.0, 1.0]",
         0.2,
         0.0,
         1.0);
   paraParams[DualBoundGainBranchRatio] = new ParaParamReal(
         "DualBoundGainBranchRatio",
         "# Branch if the dual bound gain is less than average of the that multiplied by this value.  [Default: 0.5][0.0, 3.0]",
         0.5,
         0.0,
         3.0);
   paraParams[CollectingModeInterval] = new ParaParamReal(
         "CollectingModeInterval",
         "# If the time interval between collection mode is less than this time, increase the number of collecting nodes. Negative value: no dynamic adjust. NOT set 0.0. [Default: 10.0][-1.0, DBL_MAX]",
         10.0,
         -1.0,
         DBL_MAX);
   paraParams[RestartInRampDownThresholdTime] = new ParaParamReal(
         "RestartInRampDownThresholdTime",
         "# Threshold time to restart in ramp-down phase. -1.0: no restart. Do not set a positive value less than 1.0. [Default: -1.0][-1.0, DBL_MAX]",
         -1.0,
         -1.0,
         DBL_MAX);
   paraParams[RestartInRampDownActiveSolverRatio] = new ParaParamReal(
         "RestartInRampDownActiveSolverRatio",
         "# (EXPERIMENTAL: CURRENTLY UNUSED PARAMETER) Ratio of active solvers ratio to restart in ramp-down phase. Restart if the active solvers ratio less than this value keeps until the threshold time. [Default: 0.7][0.01, 0.99]",
         0.7,
         0.01,
         0.99);
   paraParams[HugeImbalanceThresholdTime] = new ParaParamReal(
         "HugeImbalanceThresholdTime",
         "# Threshold time to detect huge imbalance. -1.0: no detect. NOT set 0.0. [Default: -1.0][-1.0, DBL_MAX]",
         -1.0,
         -1.0,
         DBL_MAX);
   paraParams[HugeImbalanceActiveSolverRatio] = new ParaParamReal(
         "HugeImbalanceActiveSolverRatio",
         "# Active solvers ratio to detect huge imbalance. If the active solvers ratio less than the value keeps until the threshold time, detect huge imbalance. [Default: 0.9][0.01, 0.99]",
         0.9,
         0.01,
         0.99);
   paraParams[TimeStopSolvingMode] = new ParaParamReal(
         "TimeStopSolvingMode",
         "# Candidate of stop solving mode: Node computing time less than this value. The value less than 0 means to not stop solving by this criterion. [Default: -1][-1.0, DBL_MAX]",
         -1.0,
         -1.0,
         DBL_MAX);
   paraParams[NoTransferThresholdReductionRatio] = new ParaParamReal(
         "NoTransferThresholdReductionRatio",
         "# Reduction ratio for threshold value of no node transfer. [Default: 1.0][0.0, 1.0]",
          1.0,
          0.0,
          1.0);
   paraParams[EnhancedCheckpointStartTime] = new ParaParamReal(
         "EnhancedCheckpointStartTime",
         "# Start time to collect open nodes for enhanced checkpoint files. [Default: 36000.0][0.0, DBL_MAX]",
          36000.0,
          0.0,
          DBL_MAX);

   /** char params */

   /** string params */
   paraParams[SolverSettingsForInitialPresolving] = new ParaParamString(
         "SolverSettingsForInitialPresolving",
         "# Solver parameter settings filename that is applied at initial presolving in LoadCoordinator. Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsAtRootNode] = new ParaParamString(
         "SolverSettingsAtRootNode",
         "# Solver parameter settings filename that is applied at root node solving (including presolving in load coordinator). Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsExceptRootNode] = new ParaParamString(
         "SolverSettingsExceptRootNode",
         "# Solver parameter settings filename that is applied at all nodes solving except root node. Empty name use default settings. [Default: ]",
         "");
   paraParams[SolverSettingsAtRacing] = new ParaParamString(
         "SolverSettingsAtRacing",
         "# Solver parameter settings filename that is applied at racing stage. Empty name use default settings. [Default: ]",
         "");
}

void
BbParaParamSet::read(
      ParaComm *comm,
      const char* filename
      )
{

   ParaParamSet::read(comm, filename);

   ///
   /// check parameter consistency
   ///
   if( getBoolParamValue(RacingStatBranching) &&
         ( ( getIntParamValue(RampUpPhaseProcess) != 1 && getIntParamValue(RampUpPhaseProcess) != 2 ) ||
               (!getBoolParamValue(CollectOnce)) ) )
   {
      std::cout << "Input error in file <" << filename
            << "> [RacingStatBranching = TRUE] needs to be [RampUpPhaseProcess = 1 or 2] and [CollectOnce = TRUE]"
            << std::endl;
      exit(1);
   }
   if( getBoolParamValue(Quiet) )
   {
      setBoolParamValue(TagTrace, false);
      setBoolParamValue(LogSolvingStatus, false);
      setBoolParamValue(LogTasksTransfer, false);
   }
   if( getIntParamValue(RampUpPhaseProcess) == 2 )
   {
      setBoolParamValue(CollectOnce, true);
      setIntParamValue(NCollectOnce, -1);    // anyway,  collects all nodes once
   }

}

